/*
 ***************************************************************************
 * Ralink Tech Inc.
 * 4F, No. 2 Technology 5th Rd.
 * Science-based Industrial Park
 * Hsin-chu, Taiwan, R.O.C.
 *
 * (c) Copyright 2002, Ralink Technology, Inc.
 *
 * All rights reserved. Ralink's source code is an unpublished work and the
 * use of a copyright notice does not imply otherwise. This source code
 * contains confidential trade secret material of Ralink Tech. Any attemp
 * or participation in deciphering, decoding, reverse engineering or in any
 * way altering the source code is stricitly prohibited, unless the prior
 * written consent of Ralink Technology, Inc. is obtained.
 ***************************************************************************

    Module Name:
    wpa.c

    Abstract:

    Revision History:
    Who         When            What
    --------    ----------      ----------------------------------------------
    Jan Lee     03-07-22        Initial
    Rory Chen   04-11-29        Add WPA2PSK
*/
#include "rt_config.h"

UCHAR       OUI_WPA_WEP40[4]        = {0x00, 0x50, 0xF2, 0x01};
UCHAR       OUI_WPA_TKIP[4]     = {0x00, 0x50, 0xF2, 0x02};
UCHAR       OUI_WPA_CCMP[4]     = {0x00, 0x50, 0xF2, 0x04};
UCHAR       OUI_WPA2_WEP40[4]   = {0x00, 0x0F, 0xAC, 0x01};
UCHAR       OUI_WPA2_TKIP[4]        = {0x00, 0x0F, 0xAC, 0x02};
UCHAR       OUI_WPA2_CCMP[4]        = {0x00, 0x0F, 0xAC, 0x04};

#define	ADD_ONE_To_64BIT_VAR(_V)		\
{										\
	UCHAR	cnt = LEN_KEY_DESC_REPLAY;	\
	do									\
	{									\
		cnt--;							\
		_V[cnt]++;						\
		if (cnt == 0)					\
			break;						\
	}while (_V[cnt] == 0);				\
}

BOOLEAN WPAMsgTypeSubst(
    IN UCHAR    EAPType,
    OUT ULONG   *MsgType)
{
    switch(EAPType)
    {
        case EAPPacket:
            *MsgType = APMT2_EAPPacket;
            break;
        case EAPOLStart:
            *MsgType = APMT2_EAPOLStart;
            break;
        case EAPOLLogoff:
            *MsgType = APMT2_EAPOLLogoff;
            break;
        case EAPOLKey:
            *MsgType = APMT2_EAPOLKey;
            break;
        case EAPOLASFAlert:
            *MsgType = APMT2_EAPOLASFAlert;
            break;
        default:
            DBGPRINT(RT_DEBUG_TRACE, ("WPAMsgTypeSubst : return FALSE; \n"));
            return FALSE;
    }

    return TRUE;
}

/*
    ==========================================================================
    Description:
        association state machine init, including state transition and timer init
    Parameters:
        S - pointer to the association state machine
    ==========================================================================
 */
VOID APWpaStateMachineInit(
    IN  PRTMP_ADAPTER   pAd,
    IN  STATE_MACHINE *S,
    OUT STATE_MACHINE_FUNC Trans[])
{
//    NDIS_STATUS Status = NDIS_STATUS_SUCCESS;

    StateMachineInit(S, Trans, AP_MAX_WPA_PTK_STATE, AP_MAX_WPA_MSG, (STATE_MACHINE_FUNC)Drop, AP_WPA_PTK, AP_WPA_MACHINE_BASE);

    StateMachineSetAction(S, AP_WPA_PTK, APMT2_EAPPacket, (STATE_MACHINE_FUNC)APWpaEAPPacketAction);
    StateMachineSetAction(S, AP_WPA_PTK, APMT2_EAPOLStart, (STATE_MACHINE_FUNC)APWpaEAPOLStartAction);
    StateMachineSetAction(S, AP_WPA_PTK, APMT2_EAPOLLogoff, (STATE_MACHINE_FUNC)APWpaEAPOLLogoffAction);
    StateMachineSetAction(S, AP_WPA_PTK, APMT2_EAPOLKey, (STATE_MACHINE_FUNC)APWpaEAPOLKeyAction);
    StateMachineSetAction(S, AP_WPA_PTK, APMT2_EAPOLASFAlert, (STATE_MACHINE_FUNC)APWpaEAPOLASFAlertAction);
}

/*
    ==========================================================================
    Description:
        this is state machine function.
        When receiving EAP packets which is  for 802.1x authentication use.
        Not use in PSK case
    Return:
    ==========================================================================
*/
VOID APWpaEAPPacketAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
}

VOID APWpaEAPOLASFAlertAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
}

VOID APWpaEAPOLLogoffAction(
    IN PRTMP_ADAPTER pAd,
    IN MLME_QUEUE_ELEM *Elem)
{
}

/*
    ==========================================================================
    Description:
        Port Access Control Inquiry function. Return entry's Privacy and Wpastate.
    Return:
        pEntry
    ==========================================================================
*/
MAC_TABLE_ENTRY *PACInquiry(
    IN  PRTMP_ADAPTER               pAd,
    IN  PUCHAR                      pAddr,
    OUT NDIS_802_11_PRIVACY_FILTER  *Privacy,
    OUT AP_WPA_STATE                *WpaState)
{
    MAC_TABLE_ENTRY *pEntry = (MAC_TABLE_ENTRY*)NULL;

    *Privacy = Ndis802_11PrivFilterAcceptAll;
    *WpaState = AS_NOTUSE;

    if (pAddr[0] & 0x01)
    {// mcast & broadcast address
        *Privacy = Ndis802_11PrivFilterAcceptAll;
        *WpaState = AS_NOTUSE;
    }
    else
    {// unicast address
        pEntry = MacTableLookup(pAd, pAddr);
        if (pEntry)
        {
            *Privacy = pEntry->PrivacyFilter;
            *WpaState = pEntry->WpaState;
        }
    }

    return pEntry;
}

/*
    ==========================================================================
    Description:

    Return:
         TRUE if this is EAP frame
         FALSE otherwise
    ==========================================================================
*/
BOOLEAN RTMPCheckWPAframe(
    IN PRTMP_ADAPTER    pAd,
    IN PUCHAR           pHeader,
    IN ULONG            DataByteCount)
{
    PUCHAR  pData;

    if(DataByteCount < (LENGTH_802_11+LENGTH_802_1_H +LENGTH_EAPOL_H))
        return FALSE;

    DBGPRINT(RT_DEBUG_INFO, ("RTMPCheckWPAframe ===> \n"));

    pData = pHeader + LENGTH_802_11;
    DataByteCount -= LENGTH_802_11;

    if (NdisEqualMemory(SNAP_802_1H, pData, 6))
    {
        pData += 6;
    }
    if (NdisEqualMemory(EAPOL, pData, 2))
    {
        pData += 2;
    }
    else
        return FALSE;

    switch (*(pData+1))
    {
        case EAPPacket:
            DBGPRINT(RT_DEBUG_TRACE, ("Receive EAP-Packet frame, TYPE = 0 \n"));
            break;
        case EAPOLStart:
            DBGPRINT(RT_DEBUG_TRACE, ("Receive EAPOL-Start frame, TYPE = 1 \n"));
            break;
        case EAPOLLogoff:
            DBGPRINT(RT_DEBUG_TRACE, ("Receive EAPOLLogoff frame, TYPE = 2 \n"));
            break;
        case EAPOLKey:
            DBGPRINT(RT_DEBUG_TRACE, ("Receive EAPOL-Key frame, TYPE = 3, Length =%x\n", *(pData+2)));
            break;
        case EAPOLASFAlert:
            DBGPRINT(RT_DEBUG_TRACE, ("Receive EAPOLASFAlert frame, TYPE = 4 \n"));
            break;
        default:
            return FALSE;

    }
    return TRUE;
}

/*
    ==========================================================================
    Description:
       Check sanity of multicast cipher selector in RSN IE.
    Return:
         TRUE if match
         FALSE otherwise
    ==========================================================================
*/
BOOLEAN RTMPCheckMcast(
    IN PRTMP_ADAPTER    pAd,
    IN PEID_STRUCT      eid_ptr,
    IN MAC_TABLE_ENTRY  *pEntry)
{
    pEntry->AuthMode = pAd->CommonCfg.AuthMode;

    if (eid_ptr->Len >= 6)
    {
        // WPA and WPA2 format not the same in RSN_IE
        if (eid_ptr->Eid == IE_WPA)
        {
            if (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA1WPA2)
                pEntry->AuthMode = Ndis802_11AuthModeWPA;
            else if (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA1PSKWPA2PSK)
                pEntry->AuthMode = Ndis802_11AuthModeWPAPSK;

            if (NdisEqualMemory(&eid_ptr->Octet[6], &pAd->ApCfg.RSN_IE[0][6], 4))
                return TRUE;
        }
        else if (eid_ptr->Eid == IE_WPA2)
        {
            UCHAR   IE_Idx = 0;

            // When WPA1/WPA2 mix mode, the RSN_IE is stored in different structure
            if ((pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA1WPA2) ||
                (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA1PSKWPA2PSK))
                IE_Idx = 1;

            if (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA1WPA2)
                pEntry->AuthMode = Ndis802_11AuthModeWPA2;
            else if (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA1PSKWPA2PSK)
                pEntry->AuthMode = Ndis802_11AuthModeWPA2PSK;

            if (NdisEqualMemory(&eid_ptr->Octet[2], &pAd->ApCfg.RSN_IE[IE_Idx][2], 4))
                return TRUE;
        }
    }

    DBGPRINT(RT_DEBUG_ERROR, ("RTMPCheckMcast ==> WPAIE = %d\n", eid_ptr->Eid));

    return FALSE;
}

/*
    ==========================================================================
    Description:
       Check sanity of unicast cipher selector in RSN IE.
    Return:
         TRUE if match
         FALSE otherwise
    ==========================================================================
*/
BOOLEAN RTMPCheckUcast(
    IN PRTMP_ADAPTER    pAd,
    IN PEID_STRUCT      eid_ptr,
    IN MAC_TABLE_ENTRY  *pEntry)
{
    pEntry->WepStatus = pAd->CommonCfg.WepStatus;

    if (eid_ptr->Len >= 16)
    {
        if (eid_ptr->Eid == IE_WPA)
        {
            if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption4Enabled)
            {// multiple cipher (TKIP/CCMP)
                if (NdisEqualMemory(&eid_ptr->Octet[12], &pAd->ApCfg.RSN_IE[0][12], 4))
                {
                    pEntry->WepStatus = Ndis802_11Encryption2Enabled;
                    return TRUE;
                }
                else if (NdisEqualMemory(&eid_ptr->Octet[12], &pAd->ApCfg.RSN_IE[0][16], 4))
                {
                    pEntry->WepStatus = Ndis802_11Encryption3Enabled;
                    return TRUE;
                }
            }
            else
            {// single cipher
                if (NdisEqualMemory(&eid_ptr->Octet[12], &pAd->ApCfg.RSN_IE[0][12], 4))
                    return TRUE;
            }
        }
        else if (eid_ptr->Eid == IE_WPA2)
        {
            UCHAR   IE_Idx = 0;

            // When WPA1/WPA2 mix mode, the RSN_IE is stored in different structure
            if ((pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA1WPA2) ||
                (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA1PSKWPA2PSK))
                IE_Idx = 1;

            if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption4Enabled)
            {// multiple cipher (TKIP/CCMP)
                if (NdisEqualMemory(&eid_ptr->Octet[8], &pAd->ApCfg.RSN_IE[IE_Idx][8], 4))
                {
                    pEntry->WepStatus = Ndis802_11Encryption2Enabled;
                    return TRUE;
                }
                else if (NdisEqualMemory(&eid_ptr->Octet[8], &pAd->ApCfg.RSN_IE[IE_Idx][12], 4))
                {
                    pEntry->WepStatus = Ndis802_11Encryption3Enabled;
                    return TRUE;
                }
            }
            else
            {// single cipher
                if (NdisEqualMemory(&eid_ptr->Octet[8], &pAd->ApCfg.RSN_IE[IE_Idx][8], 4))
                    return TRUE;
            }
        }
    }

    DBGPRINT(RT_DEBUG_ERROR, ("RTMPCheckUcast ==> WPAIE = %d\n", eid_ptr->Eid));

    return FALSE;
}

/*
    ==========================================================================
    Description:
       Check sanity of authentication method selector in RSN IE.
    Return:
         TRUE if match
         FALSE otherwise
    ==========================================================================
*/
BOOLEAN RTMPCheckAUTH(
    IN PRTMP_ADAPTER    pAd,
    IN PEID_STRUCT      eid_ptr,
    IN MAC_TABLE_ENTRY  *pEntry)
{
    if (eid_ptr->Len >= 16)
    {
        if (eid_ptr->Eid == IE_WPA)
        {
            if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption4Enabled)
            {
                if (NdisEqualMemory(&eid_ptr->Octet[18], &pAd->ApCfg.RSN_IE[0][18], 4))
                    return TRUE;
                else if (NdisEqualMemory(&eid_ptr->Octet[18], &pAd->ApCfg.RSN_IE[0][22], 4))
                    return TRUE;
            }
            else
            {
                if (NdisEqualMemory(&eid_ptr->Octet[18], &pAd->ApCfg.RSN_IE[0][18], 4))
                    return TRUE;
            }
        }
        else if (eid_ptr->Eid == IE_WPA2)
        {
            UCHAR   IE_Idx = 0;

            // When WPA1/WPA2 mix mode, the RSN_IE is stored in different structure
            if ((pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA1WPA2) ||
                (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA1PSKWPA2PSK))
                IE_Idx = 1;

            if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption4Enabled)
            {
                if (NdisEqualMemory(&eid_ptr->Octet[14], &pAd->ApCfg.RSN_IE[IE_Idx][14], 4))
                    return TRUE;
                else if (NdisEqualMemory(&eid_ptr->Octet[14], &pAd->ApCfg.RSN_IE[IE_Idx][18], 4))
                    return TRUE;
            }
            else
            {
                if (NdisEqualMemory(&eid_ptr->Octet[14], &pAd->ApCfg.RSN_IE[IE_Idx][14], 4))
                    return TRUE;
            }
        }
    }

    DBGPRINT(RT_DEBUG_ERROR, ("RTMPCheckAUTH ==> WPAIE=%d, WepStatus=%d\n", eid_ptr->Eid, pEntry->WepStatus));

    return FALSE;
}

/*
    ==========================================================================
    Description:
       Start 4-way HS when rcv EAPOL_START which may create by our driver in assoc.c
    Return:
    ==========================================================================
*/
VOID APWpaEAPOLStartAction(
    IN PRTMP_ADAPTER    pAd,
    IN MLME_QUEUE_ELEM  *Elem)
{
	MAC_TABLE_ENTRY     *pEntry;
	PHEADER_802_11      pHeader;
	BOOLEAN             Cancelled;

	DBGPRINT(RT_DEBUG_TRACE, ("APWpaEAPOLStartAction ===> \n"));

	// Delay for avoiding the misorder of EAPOL-KEY and Assoc Rsp.
	//RTMPusecDelay(200000);//200ms

	pHeader = (PHEADER_802_11)Elem->Msg;

	//For normaol PSK, we enqueue an EAPOL-Start command to trigger the process.
	if (Elem->MsgLen == 6)
		pEntry = MacTableLookup(pAd, Elem->Msg);
	else
		pEntry = MacTableLookup(pAd, pHeader->Addr2);

	if (pEntry)
	{
		if ((pEntry->PortSecured == WPA_802_1X_PORT_NOT_SECURED) && (pEntry->WpaState < AS_PTKSTART)
			&& ((pEntry->AuthMode == Ndis802_11AuthModeWPAPSK) || (pEntry->AuthMode == Ndis802_11AuthModeWPA2PSK)))
		{
			pEntry->PrivacyFilter = Ndis802_11PrivFilter8021xWEP;
			pEntry->WpaState = AS_INITPSK;
			pEntry->PortSecured = WPA_802_1X_PORT_NOT_SECURED;
			NdisZeroMemory(pEntry->R_Counter, sizeof(pEntry->R_Counter));
			pEntry->ReTryCounter = PEER_MSG1_RETRY_TIMER_CTR;

			WPAStart4WayHS(pAd, pEntry);
			NdisMCancelTimer(&pEntry->RetryTimer, &Cancelled);
			NdisMSetTimer(&pEntry->RetryTimer, PEER_MSG1_RETRY_EXEC_INTV);
		}
	}
}

/*
    ==========================================================================
    Description:
        Function to handle countermeasures active attack.  Init 60-sec timer if necessary.
    Return:
    ==========================================================================
*/
VOID HandleCounterMeasure(
    IN PRTMP_ADAPTER    pAd,
    IN MAC_TABLE_ENTRY  *pEntry)
{
    INT         i;
    BOOLEAN     Cancelled;

    if (!pEntry)
        return;

    // record which entry causes this MIC error, if this entry sends disauth/disassoc, AP doesn't need to log the CM
    pEntry->CMTimerRunning = TRUE;
    pAd->ApCfg.MICFailureCounter++;

    if (pAd->ApCfg.CMTimerRunning == TRUE)
    {
        DBGPRINT(RT_DEBUG_ERROR, ("Receive CM Attack Twice within 60 seconds ====>>> \n"));

        // renew GTK
        APGenRandom(pAd, pAd->ApCfg.GNonce);
        ApLogEvent(pAd, pEntry->Addr, EVENT_COUNTER_M);
        NdisMCancelTimer(&pAd->ApCfg.CounterMeasureTimer, &Cancelled);
        for (i = 0; i < MAX_LEN_OF_MAC_TABLE; i++)
        {
            // happened twice within 60 sec,  AP SENDS disaccociate all associated STAs.  All STA's transition to State 2
            if (pAd->MacTab.Content[i].Valid == TRUE)
            {
                DisAssocAction(pAd, &pAd->MacTab.Content[i], REASON_MIC_FAILURE);
            }
        }

        // Further,  ban all Class 3 DATA transportation for  a period 0f 60 sec
        // disallow new association , too
        pAd->ApCfg.BANClass3Data = TRUE;

        // check how many entry left...  should be zero
        pAd->ApCfg.GKeyDoneStations = pAd->MacTab.Size;
        DBGPRINT(RT_DEBUG_TRACE, ("pAd->ApCfg.GKeyDoneStations=%d \n", pAd->ApCfg.GKeyDoneStations));
    }

    NdisMSetTimer(&pAd->ApCfg.CounterMeasureTimer, 60*MLME_TASK_EXEC_INTV);
    pAd->ApCfg.CMTimerRunning = TRUE;
    pAd->ApCfg.PrevaMICFailTime = pAd->ApCfg.aMICFailTime;
    NdisGetCurrentSystemTime(&pAd->ApCfg.aMICFailTime);
}

/*
    ==========================================================================
    Description:
        This is state machine function.
        When receiving EAPOL packets which is  for 802.1x key management.
        Use both in WPA, and WPAPSK case.
        In this function, further dispatch to different functions according to the received packet.  3 categories are :
          1.  normal 4-way pairwisekey and 2-way groupkey handshake
          2.  MIC error (Countermeasures attack)  report packet from STA.
          3.  Request for pairwise/group key update from STA
    Return:
    ==========================================================================
*/
VOID APWpaEAPOLKeyAction(
    IN PRTMP_ADAPTER    pAd,
    IN MLME_QUEUE_ELEM  *Elem)
{
    BOOLEAN             Cancelled;
    MAC_TABLE_ENTRY     *pEntry;
    PHEADER_802_11      pHeader;
    PAP_EAPOL_PACKET       pEapol_packet;

    DBGPRINT(RT_DEBUG_INFO, ("APWpaEAPOLKeyAction ===>\n"));

    pHeader = (PHEADER_802_11)Elem->Msg;
    pEapol_packet = (PAP_EAPOL_PACKET)&Elem->Msg[LENGTH_802_11 + LENGTH_802_1_H];

    do
    {
        if (pAd->CommonCfg.AuthMode < Ndis802_11AuthModeWPA)
            break;

        pEntry = MacTableLookup(pAd, pHeader->Addr2);
        if (!pEntry)
            break;

        if ((pEapol_packet->ProVer != EAPOL_VER) || ((pEapol_packet->KeyDesc.Type != WPA1_KEY_DESC) && (pEapol_packet->KeyDesc.Type != WPA2_KEY_DESC)))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("Key descripter does not match with WPA rule\n"));
            break;
        }

        if ((pEntry->WepStatus == Ndis802_11Encryption2Enabled) && (pEapol_packet->KeyDesc.Keyinfo.KeyDescVer != DESC_TYPE_TKIP))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("Key descripter version not match(TKIP) \n"));
            break;
        }
        else if ((pEntry->WepStatus == Ndis802_11Encryption3Enabled) && (pEapol_packet->KeyDesc.Keyinfo.KeyDescVer != DESC_TYPE_AES))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("Key descripter version not match(AES) \n"));
            break;
        }

        if ((pEntry->Sst == SST_ASSOC) && (pEntry->WpaState >= AS_INITPSK))
        {
            if ((pEapol_packet->KeyDesc.Keyinfo.KeyMic) && (pEapol_packet->KeyDesc.Keyinfo.Request) && (pEapol_packet->KeyDesc.Keyinfo.Error))
            {
                // KEYMIC=1, REQUEST=1, ERROR=1
                DBGPRINT(RT_DEBUG_ERROR, ("MIC, REQUEST, ERROR  are all 1, active countermeasure \n"));
                HandleCounterMeasure(pAd, pEntry);
            }
            else if ((pEapol_packet->KeyDesc.Keyinfo.Secure) && !(pEapol_packet->KeyDesc.Keyinfo.Request) && !(pEapol_packet->KeyDesc.Keyinfo.Error))
            {
                // SECURE=1, REQUEST=0, ERROR=0
                if ((pEntry->AuthMode == Ndis802_11AuthModeWPA) || (pEntry->AuthMode == Ndis802_11AuthModeWPAPSK))
                    PeerGroupMsg2Action(pAd, pEntry, &Elem->Msg[LENGTH_802_11], (Elem->MsgLen - LENGTH_802_11));
                else if ((pEntry->AuthMode == Ndis802_11AuthModeWPA2) || (pEntry->AuthMode == Ndis802_11AuthModeWPA2PSK))
                    Wpa2PeerPairMsg4Action(pAd, pEntry, Elem);
            }
            else if (!(pEapol_packet->KeyDesc.Keyinfo.Secure) && !(pEapol_packet->KeyDesc.Keyinfo.Request) && !(pEapol_packet->KeyDesc.Keyinfo.Error))
            {
                // SECURE=0, REQUEST=0, ERROR=0
                if (pEntry->WpaState == AS_PTKSTART)
                    PeerPairMsg2Action(pAd, pEntry, Elem);
                else if (pEntry->WpaState == AS_PTKINIT_NEGOTIATING)
                    Wpa1PeerPairMsg4Action(pAd, pEntry, Elem);
            }
            else if ((pEapol_packet->KeyDesc.Keyinfo.Request) && !(pEapol_packet->KeyDesc.Keyinfo.Error))
            {
                // REQUEST=1, ERROR=0
                // Need to check KeyType for groupkey or pairwise key update, refer to 8021i P.114,
                if (pEapol_packet->KeyDesc.Keyinfo.KeyType == GROUPKEY)
                {
                    DBGPRINT(RT_DEBUG_TRACE, ("REQUEST=1, ERROR=0, update group key\n"));
                    APGenRandom(pAd, pAd->ApCfg.GNonce);
                    pAd->ApCfg.WpaGTKState = SETKEYS;
                    pAd->ApCfg.GKeyDoneStations = pAd->MacTab.Size;

                    if (pEntry->WpaState == AS_PTKINITDONE)
                    {
                        pEntry->GTKState = REKEY_NEGOTIATING;
                        WPAHardTransmit(pAd, pEntry);
                        NdisMCancelTimer(&pEntry->RetryTimer, &Cancelled);
                        NdisMSetTimer(&pEntry->RetryTimer, PEER_MSG3_RETRY_EXEC_INTV);
                    }
                }
                else
                {
                    DBGPRINT(RT_DEBUG_TRACE, ("REQUEST=1, ERROR= 0, update pairwise key\n"));
                    pEntry->PairwiseKey.KeyLen = 0;
                    pEntry->PairwiseKey.CipherAlg = CIPHER_NONE;
                    AsicRemovePairwiseKeyEntry(pAd, (UCHAR)pEntry->Aid); //(pEntry - &pAd->MacTab.Content[0])/sizeof(MAC_TABLE_ENTRY));
                    pEntry->Sst = SST_ASSOC;
                    if (pEntry->AuthMode == Ndis802_11AuthModeWPA)
                        pEntry->WpaState = AS_INITPMK;
                    else if (pEntry->AuthMode == Ndis802_11AuthModeWPAPSK)
                        pEntry->WpaState = AS_INITPSK;
                    pEntry->GTKState = REKEY_NEGOTIATING;
                    pEntry->ReTryCounter = PEER_MSG1_RETRY_TIMER_CTR;
                }
            }
            else
            {
                //
            }
        }
    }while(FALSE);
}

/*
    ==========================================================================
    Description:
        This is a function to initilize 4-way handshake
    Return:

    ==========================================================================
*/
VOID WPAStart4WayHS(
    IN PRTMP_ADAPTER    pAd,
    IN MAC_TABLE_ENTRY  *pEntry)
{
//    UINT            i;
    UCHAR           Header802_3[14];
    AP_EAPOL_PACKET Packet;

    DBGPRINT(RT_DEBUG_TRACE, ("===> WPAStart4WayHS\n"));

    do
    {
        if ((!pEntry) || (!pEntry->Valid))
            break;

        if ((pEntry->WpaState > AS_PTKSTART) || (pEntry->WpaState < AS_INITPMK))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("Not expect calling  WPAStart4WayHS \n"));
            break;
        }

        // init header and Fill Packet
        MAKE_802_3_HEADER(Header802_3, pEntry->Addr, pAd->CurrentAddress, EAPOL);

        NdisZeroMemory(&Packet, sizeof(Packet));
        Packet.ProVer = EAPOL_VER;
        Packet.ProType = EAPOLKey;
        Packet.Body_Len[1] = 0x5f;

        // Increment replay counter by 1 and fill replay counter
        ADD_ONE_To_64BIT_VAR(pEntry->R_Counter);
#if 0
        i = LEN_KEY_DESC_REPLAY;
        do
        {
            i--;
            pEntry->R_Counter[i]++;
            if (i == 0)
                break;
        } while (pEntry->R_Counter[i] == 0);
#endif
        NdisMoveMemory(Packet.KeyDesc.RCounter, pEntry->R_Counter, sizeof(pEntry->R_Counter));

        // Fill key_version, keyinfo, key_len...
        Packet.KeyDesc.Keyinfo.KeyDescVer = DESC_TYPE_TKIP;
        Packet.KeyDesc.Keyinfo.KeyType = PAIRWISEKEY;
        Packet.KeyDesc.Keyinfo.KeyAck = 1;

        // Deal with the differences between WPA and WPA2
        if ((pEntry->AuthMode == Ndis802_11AuthModeWPA) || (pEntry->AuthMode == Ndis802_11AuthModeWPAPSK))
        {
            Packet.KeyDesc.Type = WPA1_KEY_DESC;
            Packet.KeyDesc.DataLen[1] = 0;
        }
        else if ((pEntry->AuthMode == Ndis802_11AuthModeWPA2) || (pEntry->AuthMode == Ndis802_11AuthModeWPA2PSK))
        {
            Packet.KeyDesc.Type = WPA2_KEY_DESC;
            Packet.KeyDesc.DataLen[1] = 0;
        }

        // Deal with the differences between TKIP and CCMP
        if (pEntry->WepStatus == Ndis802_11Encryption2Enabled)
        {
            Packet.KeyDesc.KeyLength[1] = LEN_TKIP_KEY;
            Packet.KeyDesc.Keyinfo.KeyDescVer = DESC_TYPE_TKIP;
        }
        else if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
        {
            Packet.KeyDesc.KeyLength[1] = LEN_AES_KEY;
            Packet.KeyDesc.Keyinfo.KeyDescVer = DESC_TYPE_AES;
        }

        // Fill Anonce
        APGenRandom(pAd, pEntry->ANonce);
        NdisMoveMemory(Packet.KeyDesc.Nonce, pEntry->ANonce, sizeof(pEntry->ANonce));

        // Transmit Msg1 to air
        APToWirelessSta(pAd, pEntry->Aid, Header802_3, sizeof(Header802_3), (PUCHAR)&Packet, Packet.Body_Len[1] + 4);

        pEntry->WpaState = AS_PTKSTART;
    }while(FALSE);

    DBGPRINT(RT_DEBUG_TRACE, ("<=== WPAStart4WayHS, WpaState= %d \n", pEntry->WpaState));
}

/*
    ==========================================================================
    Description:
        When receiving the second packet of 4-way pairwisekey handshake.
    Return:
    ==========================================================================
*/
VOID PeerPairMsg2Action(
    IN PRTMP_ADAPTER    pAd,
    IN MAC_TABLE_ENTRY  *pEntry,
    IN MLME_QUEUE_ELEM  *Elem)
{
    UCHAR               *PTK, *mac, *SNonce, *digest;
//    UCHAR               EAPHEAD[8] = {0xaa, 0xaa, 0x03, 0x00, 0x00, 0x00,0x88,0x8e};
    UCHAR               EAPOLHEAD[4] = {EAPOL_VER,  EAPOLKey,  0x00,  0x77};
    ULONG               FrameLen = 0;
    UCHAR               *OutBuffer;
    USHORT              MICMsgLen,MICOffset;
    BOOLEAN             Cancelled;
    PHEADER_802_11      pHeader;
    AP_EAPOL_PACKET     *pEapol_Packet, Packet;
    UCHAR               Header802_3[14];
	UCHAR				*mpool = kmalloc(80 + LEN_KEY_DESC_MIC + LEN_KEY_DESC_NONCE + 80 + MAX_LEN_OF_EAP_HS + 32, GFP_ATOMIC);

    DBGPRINT(RT_DEBUG_TRACE, ("===>PeerPairMsg2Action \n"));

	if(mpool == NULL)
		return;

	/* PTK  Len = 80 */
	PTK = (UCHAR *) ROUND_UP(mpool, 4);
	/* mac Len = LEN_KEY_DESC_MIC */
	mac = (UCHAR *) ROUND_UP(PTK + 80, 4);
	/* SNonce Len = LEN_KEY_DESC_NONCE */
	SNonce = (UCHAR *) ROUND_UP(mac + LEN_KEY_DESC_MIC, 4);
	/* digest Len = 80 */
	digest = (UCHAR *) ROUND_UP(SNonce + LEN_KEY_DESC_NONCE, 4);
	/* OutBuffer Len = MAX_LEN_OF_EAP_HS */
	OutBuffer = (UCHAR *) ROUND_UP(digest + 80, 4);

    pHeader = (PHEADER_802_11)Elem->Msg;

    do
    {
        if ((!pEntry) || (!pEntry->Valid))
            break;

        if (Elem->MsgLen < (LENGTH_802_11 + LENGTH_802_1_H + LENGTH_EAPOL_H + sizeof(AP_KEY_DESCRIPTER) - AP_MAX_LEN_OF_RSNIE - 2))
            break;

        // check Entry in valid State
        if (pEntry->WpaState < AS_PTKSTART)
            break;

        // Save Data Length to pDesc for receiving packet,   then put in outgoing frame Data Len fields.
        pEapol_Packet = (PAP_EAPOL_PACKET)&Elem->Msg[LENGTH_802_11 + LENGTH_802_1_H];

        // 802.1i, p.99 - 100 ,  on Receive MSG2
        //  1. check Replay Counter coresponds to MSG1.,  otherwise discard
        if (!NdisEqualMemory(pEapol_Packet->KeyDesc.RCounter, pEntry->R_Counter, LEN_KEY_DESC_REPLAY))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("Replay Counter Different in msg 2 of 4-way handshake! \n"));
            break;
        }

        //  2. Derive PTK,
        MICMsgLen=(Elem->Msg[LENGTH_802_11+LENGTH_802_1_H+3]) | (Elem->Msg[LENGTH_802_11+LENGTH_802_1_H+2]<<8);
        EAPOLHEAD[3]=(UCHAR)MICMsgLen;
        MICMsgLen+=LENGTH_EAPOL_H;
        NdisMoveMemory(SNonce, pEapol_Packet->KeyDesc.Nonce, LEN_KEY_DESC_NONCE);
        CountPTK(pAd->ApCfg.PMK,  pEntry->ANonce, pAd->CurrentAddress, SNonce, pHeader->Addr2,  PTK, LEN_PTK);
        NdisMoveMemory(pEntry->PTK, PTK, LEN_PTK);

        //  3. verify MSG2 MIC, If not valid, discard.
        MICOffset = LENGTH_EAPOL_H+sizeof(AP_KEY_DESCRIPTER)-AP_MAX_LEN_OF_RSNIE-2-LEN_KEY_DESC_MIC;
        NdisMoveMemory(OutBuffer, &Elem->Msg[LENGTH_802_11+LENGTH_802_1_H], MICMsgLen);
        NdisZeroMemory((OutBuffer+MICOffset), LEN_KEY_DESC_MIC);

        if (pEntry->WepStatus == Ndis802_11Encryption2Enabled)
        {
            hmac_md5(PTK, LEN_EAP_MICK, OutBuffer, MICMsgLen, mac);
        }
        else if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
        {
            HMAC_SHA1(OutBuffer,  MICMsgLen, PTK, LEN_EAP_MICK, digest);
            NdisMoveMemory(mac, digest, LEN_KEY_DESC_MIC);
        }

        if (!NdisEqualMemory(&Elem->Msg[LENGTH_802_11+LENGTH_802_1_H+MICOffset], mac, LEN_KEY_DESC_MIC))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("MIC Different in msg 2 of 4-way handshake\n"));
            DBGPRINT(RT_DEBUG_ERROR, ("Receive : %x %x \n ",Elem->Msg[LENGTH_802_11+LENGTH_802_1_H+MICOffset], Elem->Msg[LENGTH_EAPOL_H+LENGTH_802_11+LENGTH_802_1_H+MICOffset+1]));
            break;
        }
        else
            DBGPRINT(RT_DEBUG_ERROR, ("MIC VALID in msg 2 of 4-way handshake\n"));

        //  4. check RSN IE, if not match, send MLME-DEAUTHENTICATE.
        // Points  pDesc  to RSN_IE in Datafield.
        if (!NdisEqualMemory(&pEapol_Packet->KeyDesc.Data[2], pEntry->RSN_IE, pEntry->RSNIE_Len))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("PeerPairMsg2Action::RSN_IE Different in msg 2 of 4-way handshake!\n"));
            DBGPRINT(RT_DEBUG_TRACE, ("PeerPairMsg2Action::Receive     : %x %x %x %x \n", pEapol_Packet->KeyDesc.Data[2], pEapol_Packet->KeyDesc.Data[3], pEapol_Packet->KeyDesc.Data[4], pEapol_Packet->KeyDesc.Data[5]));
            DBGPRINT(RT_DEBUG_TRACE, ("PeerPairMsg2Action::Current(%d) : %x %x %x %x \n", pEntry->RSNIE_Len, pEntry->RSN_IE[0], pEntry->RSN_IE[1], pEntry->RSN_IE[2], pEntry->RSN_IE[3]));
            break;
        }
        else
            DBGPRINT(RT_DEBUG_TRACE, ("RSN_IE VALID in msg 2 of 4-way handshake\n"));

        //verified  ok , change state
        NdisMCancelTimer(&pEntry->RetryTimer, &Cancelled);
        pEntry->WpaState = AS_PTKINIT_NEGOTIATING;

        // Increment replay counter by 1
        ADD_ONE_To_64BIT_VAR(pEntry->R_Counter);

        // CONSTRUCT msg3
        // Init Packet and Fill header
        NdisZeroMemory(&Packet, sizeof(Packet));
        Packet.ProVer = EAPOL_VER;
        Packet.ProType = EAPOLKey;

        // Fill replay counter
        NdisMoveMemory(Packet.KeyDesc.RCounter, pEntry->R_Counter, sizeof(pEntry->R_Counter));

        // Fill key version, keyinfo, key len
        Packet.KeyDesc.Keyinfo.KeyMic = 1;
        Packet.KeyDesc.Keyinfo.KeyType = PAIRWISEKEY;
        Packet.KeyDesc.Keyinfo.Install = 1;
        Packet.KeyDesc.Keyinfo.KeyAck = 1;
        if ((pEntry->AuthMode == Ndis802_11AuthModeWPA) || (pEntry->AuthMode == Ndis802_11AuthModeWPAPSK))
        {
            Packet.KeyDesc.Type = WPA1_KEY_DESC;
            Packet.KeyDesc.DataLen[1] = 0;
        }
        else if ((pEntry->AuthMode == Ndis802_11AuthModeWPA2) || (pEntry->AuthMode == Ndis802_11AuthModeWPA2PSK))
        {
            Packet.KeyDesc.Type = WPA2_KEY_DESC;
            Packet.KeyDesc.DataLen[1] = 0;
            Packet.KeyDesc.Keyinfo.EKD_DL = 1;
            Packet.KeyDesc.Keyinfo.Secure = 1;
        }

        if (pEntry->WepStatus == Ndis802_11Encryption2Enabled)
        {
            Packet.KeyDesc.KeyLength[1] = LEN_TKIP_KEY;
        }
        else if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
        {
            Packet.KeyDesc.KeyLength[1] = LEN_AES_KEY;
        }
        // Need to care about TKIP/AES both cipher mode carefully.
        Packet.KeyDesc.Keyinfo.KeyDescVer =
        	(((pEntry->WepStatus == Ndis802_11Encryption3Enabled) || (pAd->CommonCfg.GroupKeyWepStatus == Ndis802_11Encryption3Enabled)) ? (DESC_TYPE_AES) : (DESC_TYPE_TKIP));

        // Fill in Data, Data Len
        NdisMoveMemory(Packet.KeyDesc.Nonce, pEntry->ANonce, sizeof(pEntry->ANonce));
        NdisZeroMemory(Packet.KeyDesc.MIC, sizeof(Packet.KeyDesc.MIC));

        if ((pEntry->AuthMode == Ndis802_11AuthModeWPA) || (pEntry->AuthMode == Ndis802_11AuthModeWPAPSK))
        {
            Packet.KeyDesc.Data[0] = IE_WPA;
            Packet.KeyDesc.Data[1] = pAd->ApCfg.RSNIE_Len[0];
            Packet.KeyDesc.DataLen[1] = pAd->ApCfg.RSNIE_Len[0] + 2;
            Packet.Body_Len[1] = 93 + 2 + 2 + pAd->ApCfg.RSNIE_Len[0];

            NdisMoveMemory(&Packet.KeyDesc.Data[2], pAd->ApCfg.RSN_IE[0], pAd->ApCfg.RSNIE_Len[0]);

            // make a frame for Counting MIC.
            MakeOutgoingFrame(OutBuffer,            &FrameLen,
                                Packet.Body_Len[1] + 4,  &Packet,
                                END_OF_ARGS);
            NdisZeroMemory(mac, sizeof(mac));

            // count MIC
            if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
            {
                HMAC_SHA1(OutBuffer,  FrameLen, PTK, LEN_EAP_MICK, digest);
                NdisMoveMemory(mac, digest, LEN_KEY_DESC_MIC);
            }
            else
            {
                hmac_md5(PTK,  LEN_EAP_MICK, OutBuffer, FrameLen, mac);
            }
            NdisMoveMemory(&Packet.KeyDesc.MIC, mac, LEN_KEY_DESC_MIC);
        }
        else if ((pEntry->AuthMode == Ndis802_11AuthModeWPA2) || (pEntry->AuthMode == Ndis802_11AuthModeWPA2PSK))
        {
            UCHAR       GTK[TKIP_GTK_LENGTH],pRc4GTK[512], mic[LEN_KEY_DESC_MIC];
            UCHAR       ekey[(LEN_KEY_DESC_IV+LEN_EAP_EK)], Key_Data[512], key_length, RSNIE_Len;

            NdisMoveMemory(Packet.KeyDesc.RSC, pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxTsc, 6);

            CountGTK(pAd->ApCfg.GMK, (UCHAR*)pAd->ApCfg.GNonce, pAd->CurrentAddress, GTK, TKIP_GTK_LENGTH);
            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].KeyLen = LEN_TKIP_EK;
            NdisMoveMemory(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].Key, GTK, LEN_TKIP_EK);
            NdisMoveMemory(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxMic, &GTK[16], LEN_TKIP_TXMICK);
            NdisMoveMemory(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].RxMic, &GTK[24], LEN_TKIP_RXMICK);

            if (pEntry->WepStatus == Ndis802_11Encryption2Enabled)
                pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg = CIPHER_TKIP;
            else if (pAd->CommonCfg.WepStatus == Ndis802_11Encryption4Enabled)
                pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg = CIPHER_TKIP;
            else if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
                pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg = CIPHER_AES;
            else
                pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg = CIPHER_NONE;

            // install Group Key to MAC ASIC
            AsicAddSharedKeyEntry(
                pAd,
                BSS0,
                pAd->CommonCfg.DefaultKeyId,
                pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg,
                pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].Key,
                MAX_LEN_OF_SHARE_KEY,
                pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxMic,
                pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].RxMic);

            NdisZeroMemory(Key_Data, sizeof(Key_Data));

            if (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA1PSKWPA2PSK)
            {
                RSNIE_Len = pAd->ApCfg.RSNIE_Len[1];
                NdisMoveMemory(&Key_Data[2], pAd->ApCfg.RSN_IE[1], RSNIE_Len);
            }
            else
            {
                RSNIE_Len = pAd->ApCfg.RSNIE_Len[0];
                NdisMoveMemory(&Key_Data[2], &pAd->ApCfg.RSN_IE[0], RSNIE_Len);
            }

            Packet.Body_Len[1] = 2+RSNIE_Len;

            Key_Data[0] = IE_WPA2;
            Key_Data[1] = RSNIE_Len;

            // Key Data Encapsulation format
            Key_Data[2+RSNIE_Len+0] = 0xDD;
            Key_Data[2+RSNIE_Len+2] = 0x00;
            Key_Data[2+RSNIE_Len+3] = 0x0F;
            Key_Data[2+RSNIE_Len+4] = 0xAC;
            Key_Data[2+RSNIE_Len+5] = 0x01;

            // GTK Key Data Encapsulation format
            Key_Data[2+RSNIE_Len+6] = (pAd->CommonCfg.DefaultKeyId & 0x03);
            Key_Data[2+RSNIE_Len+7] = 0x00;

            // handle the difference between TKIP and AES-CCMP
            if (pAd->CommonCfg.GroupKeyWepStatus == Ndis802_11Encryption3Enabled)
            {
                Key_Data[2+RSNIE_Len+1] = 0x16;// 4+2+16(OUI+GTK+GTKKEY)
                NdisMoveMemory(&Key_Data[2+RSNIE_Len+8], GTK, LEN_AES_KEY);
	        	Packet.Body_Len[1] += 8+LEN_AES_KEY;
            }
            else
            {
                Key_Data[2+RSNIE_Len+1] = 0x26;// 4+2+32(OUI+GTK+GTKKEY)
                NdisMoveMemory(&Key_Data[2+RSNIE_Len+8], GTK, TKIP_GTK_LENGTH);
                Packet.Body_Len[1] += 8+TKIP_GTK_LENGTH;
            }

            // Still dont know why, but if not append will occur "GTK not include in MSG3"
			// Patch for compatibility between zero config and funk
			if (pAd->CommonCfg.GroupKeyWepStatus == Ndis802_11Encryption3Enabled)
			{
				Key_Data[2+RSNIE_Len+8+TKIP_GTK_LENGTH] = 0xDD;
				Key_Data[2+RSNIE_Len+8+TKIP_GTK_LENGTH+1] = 0;
				Packet.Body_Len[1] += 2;
			}
			else
			{
				Key_Data[2+RSNIE_Len+8+TKIP_GTK_LENGTH] = 0xDD;
				Key_Data[2+RSNIE_Len+8+TKIP_GTK_LENGTH+1] = 0;
				Key_Data[2+RSNIE_Len+8+TKIP_GTK_LENGTH+2] = 0;
				Key_Data[2+RSNIE_Len+8+TKIP_GTK_LENGTH+3] = 0;
				Key_Data[2+RSNIE_Len+8+TKIP_GTK_LENGTH+4] = 0;
				Key_Data[2+RSNIE_Len+8+TKIP_GTK_LENGTH+5] = 0;
				Packet.Body_Len[1] += 6;
			}

            key_length = Packet.Body_Len[1];
            Packet.KeyDesc.DataLen[1] = key_length;
            Packet.Body_Len[1] += 93 + 2;

            NdisZeroMemory(mic, sizeof(mic));

            // count MIC
            if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
            {
                AES_GTK_KEY_WRAP(&pEntry->PTK[16], Key_Data, key_length, pRc4GTK);
                // AES wrap function will grow 8 bytes in length
                NdisMoveMemory(Packet.KeyDesc.Data, pRc4GTK, (key_length + 8));
                Packet.Body_Len[1] += 8;
                Packet.KeyDesc.DataLen[1] += 8;
                MakeOutgoingFrame(OutBuffer,                &FrameLen,
                                  Packet.Body_Len[1] + 4,   &Packet,
                                  END_OF_ARGS);

                HMAC_SHA1(OutBuffer,  FrameLen, pEntry->PTK, LEN_EAP_MICK, digest);
                NdisMoveMemory(Packet.KeyDesc.MIC, digest, LEN_KEY_DESC_MIC);
            }
            else
            {
                // PREPARE Encrypted  "Key DATA" field.  (Encrypt GTK with RC4, usinf PTK[16]->[31] as Key, IV-field as IV)
                // put TxTsc in Key RSC field
                pAd->PrivateInfo.FCSCRC32 = PPPINITFCS32;   //Init crc32.

                // ekey is the contanetion of IV-field, and PTK[16]->PTK[31]
                NdisMoveMemory(ekey, Packet.KeyDesc.IV, LEN_KEY_DESC_IV);
                NdisMoveMemory(&ekey[LEN_KEY_DESC_IV], &pEntry->PTK[16], LEN_EAP_EK);
                ARCFOUR_INIT(&pAd->PrivateInfo.WEPCONTEXT, ekey, sizeof(ekey));  //INIT SBOX, KEYLEN+3(IV)
                pAd->PrivateInfo.FCSCRC32 = RTMP_CALC_FCS32(pAd->PrivateInfo.FCSCRC32, Key_Data, key_length);
                WPAARCFOUR_ENCRYPT(&pAd->PrivateInfo.WEPCONTEXT, pRc4GTK,Key_Data, key_length);
                NdisMoveMemory(Packet.KeyDesc.Data, pRc4GTK, key_length);

                // make a frame for Countint MIC,
                MakeOutgoingFrame(OutBuffer,                &FrameLen,
                                  Packet.Body_Len[1] + 4,   &Packet,
                                  END_OF_ARGS);

                hmac_md5(pEntry->PTK, LEN_EAP_MICK, OutBuffer, FrameLen, mic);
                NdisMoveMemory(Packet.KeyDesc.MIC, mic, LEN_KEY_DESC_MIC);
            }
        }

        // Make outgoing frame
        MAKE_802_3_HEADER(Header802_3, pEntry->Addr, pAd->CurrentAddress, EAPOL);

        APToWirelessSta(pAd, pEntry->Aid, Header802_3, sizeof(Header802_3), (PUCHAR)&Packet, Packet.Body_Len[1] + 4);

        pEntry->ReTryCounter = PEER_MSG3_RETRY_TIMER_CTR;
        NdisMSetTimer(&pEntry->RetryTimer, PEER_MSG3_RETRY_EXEC_INTV);

        pEntry->WpaState = AS_PTKINIT_NEGOTIATING;
    }while(FALSE);

    DBGPRINT(RT_DEBUG_TRACE, ("<=== PeerPairMsg2Action \n"));

	kfree(mpool);
	return;
}

/*
    ==========================================================================
    Description:
        countermeasures active attack timer execution
    Return:
    ==========================================================================
*/
VOID CMTimerExec(
    IN PVOID SystemSpecific1,
    IN PVOID FunctionContext,
    IN PVOID SystemSpecific2,
    IN PVOID SystemSpecific3)
{
    UINT            i,j=0;
    PRTMP_ADAPTER   pAd = (PRTMP_ADAPTER)FunctionContext;

    pAd->ApCfg.BANClass3Data = FALSE;
    for (i = 0; i < MAX_LEN_OF_MAC_TABLE; i++)
    {
        if ((pAd->MacTab.Content[i].Valid == TRUE) && (pAd->MacTab.Content[i].CMTimerRunning == TRUE))
        {
            pAd->MacTab.Content[i].CMTimerRunning =FALSE;
            j++;
        }
    }
    if (j > 1)
        DBGPRINT(RT_DEBUG_ERROR, ("Find more than one entry which generated MIC Fail ..  \n"));

    pAd->ApCfg.CMTimerRunning = FALSE;
}

VOID WPARetryProc(
    IN MAC_TABLE_ENTRY     *pEntry )
{
    if ((pEntry) && (pEntry->Valid == TRUE))
    {
        PRTMP_ADAPTER pAd = (PRTMP_ADAPTER)pEntry->pAd;

        pEntry->ReTryCounter++;
        DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec---> ReTryCounter=%d, WpaState=%d \n", pEntry->ReTryCounter, pEntry->WpaState));

        switch (pEntry->AuthMode)
        {
            case Ndis802_11AuthModeWPAPSK:
            case Ndis802_11AuthModeWPA2PSK:
                if (pEntry->ReTryCounter > (GROUP_MSG1_RETRY_TIMER_CTR + 1))
                {
                    DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec::Group Key HS exceed retry count, Disassociate client, pEntry->ReTryCounter %d\n", pEntry->ReTryCounter));
                    DisAssocAction(pAd, pEntry, REASON_GROUP_KEY_HS_TIMEOUT);
                }
                else if (pEntry->ReTryCounter > GROUP_MSG1_RETRY_TIMER_CTR)
                {
                    DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec::ReTry 2-way group-key Handshake \n"));
                    if (pEntry->GTKState == REKEY_NEGOTIATING)
                    {
                        WPAHardTransmit(pAd, pEntry);
                        NdisMSetTimer(&pEntry->RetryTimer, PEER_MSG3_RETRY_EXEC_INTV);
                    }
                }
                else if (pEntry->ReTryCounter > (PEER_MSG1_RETRY_TIMER_CTR + 3))
                {
                    DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec::MSG3 timeout, pEntry->ReTryCounter = %d\n", pEntry->ReTryCounter));
                    DisAssocAction(pAd, pEntry, REASON_4_WAY_TIMEOUT);
                }
                else if (pEntry->ReTryCounter == (PEER_MSG1_RETRY_TIMER_CTR + 3))
                {
                    DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec::Retry MSG1, the last try\n"));
                    WPAStart4WayHS(pAd , pEntry);
                    NdisMSetTimer(&pEntry->RetryTimer, PEER_MSG3_RETRY_EXEC_INTV);
                }
                else if (pEntry->ReTryCounter < (PEER_MSG1_RETRY_TIMER_CTR + 3))
                {
                    if ((pEntry->WpaState == AS_PTKSTART) || (pEntry->WpaState == AS_INITPSK) || (pEntry->WpaState == AS_INITPMK))
                    {
                        DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec::ReTry MSG1 of 4-way Handshake\n"));
                        WPAStart4WayHS(pAd, pEntry);
                        NdisMSetTimer(&pEntry->RetryTimer, PEER_MSG1_RETRY_EXEC_INTV);
                    }
                }
                break;

            default:
                break;
        }
    }

}

VOID EnqueueStartForPSKExec(
	IN PVOID SystemSpecific1,
	IN PVOID FunctionContext,
	IN PVOID SystemSpecific2,
	IN PVOID SystemSpecific3)
{
    MAC_TABLE_ENTRY *pEntry = (MAC_TABLE_ENTRY *)FunctionContext;

	if ((pEntry) && (pEntry->EnqueueStartForPSKTimerRunning == TRUE))
	{
		RTMP_ADAPTER *pAd = (PRTMP_ADAPTER)pEntry->pAd;

		pEntry->EnqueueStartForPSKTimerRunning = FALSE;
		DBGPRINT(RT_DEBUG_TRACE, ("Execute enqueue EAPoL-Start for %02x:%02x:%02x:%02x:%02x:%02x \n",
			pEntry->Addr[0], pEntry->Addr[1], pEntry->Addr[2], pEntry->Addr[3],
			pEntry->Addr[4], pEntry->Addr[5]));

		MlmeEnqueue(pAd, AP_WPA_STATE_MACHINE, APMT2_EAPOLStart, 6, &pEntry->Addr);
	}
}

VOID WPARetryExec(
    IN PVOID SystemSpecific1,
    IN PVOID FunctionContext,
    IN PVOID SystemSpecific2,
    IN PVOID SystemSpecific3)
{
    MAC_TABLE_ENTRY     *pEntry = (MAC_TABLE_ENTRY *)FunctionContext;

    if ((pEntry) && (pEntry->Valid == TRUE))
    {
        PRTMP_ADAPTER pAd = (PRTMP_ADAPTER)pEntry->pAd;

        pEntry->ReTryCounter++;
        DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec---> ReTryCounter=%d, WpaState=%d \n", pEntry->ReTryCounter, pEntry->WpaState));

        switch (pEntry->AuthMode)
        {
            case Ndis802_11AuthModeWPAPSK:
            case Ndis802_11AuthModeWPA2PSK:
                if (pEntry->ReTryCounter > (GROUP_MSG1_RETRY_TIMER_CTR + 1))
                {
                    DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec::Group Key HS exceed retry count, Disassociate client, pEntry->ReTryCounter %d\n", pEntry->ReTryCounter));
                    DisAssocAction(pAd, pEntry, REASON_GROUP_KEY_HS_TIMEOUT);
                }
                else if (pEntry->ReTryCounter > GROUP_MSG1_RETRY_TIMER_CTR)
                {
                    DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec::ReTry 2-way group-key Handshake \n"));
                    if (pEntry->GTKState == REKEY_NEGOTIATING)
                    {
                        WPAHardTransmit(pAd, pEntry);
                        RTMPModTimer(&pEntry->RetryTimer, PEER_MSG3_RETRY_EXEC_INTV);
                    }
                }
                else if (pEntry->ReTryCounter > (PEER_MSG1_RETRY_TIMER_CTR + 3))
                {
                    DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec::MSG3 timeout, pEntry->ReTryCounter = %d\n", pEntry->ReTryCounter));
                    DisAssocAction(pAd, pEntry, REASON_4_WAY_TIMEOUT);
                }
                else if (pEntry->ReTryCounter == (PEER_MSG1_RETRY_TIMER_CTR + 3))
                {
                    DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec::Retry MSG1, the last try\n"));
                    WPAStart4WayHS(pAd , pEntry);
                    RTMPModTimer(&pEntry->RetryTimer, PEER_MSG3_RETRY_EXEC_INTV);
                }
                else if (pEntry->ReTryCounter < (PEER_MSG1_RETRY_TIMER_CTR + 3))
                {
                    if ((pEntry->WpaState == AS_PTKSTART) || (pEntry->WpaState == AS_INITPSK) || (pEntry->WpaState == AS_INITPMK))
                    {
                        DBGPRINT(RT_DEBUG_TRACE, ("WPARetryExec::ReTry MSG1 of 4-way Handshake\n"));
                        WPAStart4WayHS(pAd, pEntry);
                        RTMPModTimer(&pEntry->RetryTimer, PEER_MSG1_RETRY_EXEC_INTV);
                    }
                }
                break;

            default:
                break;
        }
    }
}

/*
    ==========================================================================
    Description:
        When receiving the last packet of 4-way pairwisekey handshake.
        Initilize 2-way groupkey handshake following.
    Return:
    ==========================================================================
*/
VOID Wpa1PeerPairMsg4Action(
    IN PRTMP_ADAPTER    pAd,
    IN MAC_TABLE_ENTRY  *pEntry,
    IN MLME_QUEUE_ELEM  *Elem)
{
    UCHAR               mac[LEN_KEY_DESC_MIC];
    ULONG               MICMsgLen;
    PHEADER_802_11      pHeader;
    AP_KEY_DESCRIPTER      *pKeyDesc;
    AP_EAPOL_PACKET        EAPOLPKT;
    BOOLEAN             Cancelled;

    DBGPRINT(RT_DEBUG_TRACE, ("===> Wpa1PeerPairMsg4Action\n"));
    NdisZeroMemory((PUCHAR)&EAPOLPKT, sizeof(AP_EAPOL_PACKET));

    do
    {
        if ((!pEntry) || (!pEntry->Valid))
            break;
        if (Elem->MsgLen < (LENGTH_802_11 + LENGTH_802_1_H + LENGTH_EAPOL_H + sizeof(AP_KEY_DESCRIPTER) - AP_MAX_LEN_OF_RSNIE - 2 ) )
            break;
        if (pEntry->WpaState < AS_PTKINIT_NEGOTIATING)
            break;

        pKeyDesc = (PAP_KEY_DESCRIPTER)&Elem->Msg[(LENGTH_802_11+LENGTH_802_1_H+LENGTH_EAPOL_H)];
        pHeader = (PHEADER_802_11)Elem->Msg;

        // 1.check Replay Counter
        if (!NdisEqualMemory(pKeyDesc->RCounter, pEntry->R_Counter, LEN_KEY_DESC_REPLAY))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("Replay Counter Different in msg 4 of 4-way handshake!!!!!!!!!! \n"));
            DBGPRINT(RT_DEBUG_ERROR, ("Receive : %d %d %d %d\n", pKeyDesc->RCounter[0],pKeyDesc->RCounter[1],pKeyDesc->RCounter[2],pKeyDesc->RCounter[3]));
            DBGPRINT(RT_DEBUG_ERROR, ("Current : %d %d %d %d\n", pEntry->R_Counter[4],pEntry->R_Counter[5],pEntry->R_Counter[6],pEntry->R_Counter[7]));
        }

        // 2. check MIC, if not valid, discard silently
        MICMsgLen = (Elem->Msg[LENGTH_802_11+LENGTH_802_1_H+3]) | (Elem->Msg[LENGTH_802_11+LENGTH_802_1_H+2]<<8);
        MICMsgLen += LENGTH_EAPOL_H;

        NdisMoveMemory((PUCHAR)&EAPOLPKT, &Elem->Msg[LENGTH_802_11+LENGTH_802_1_H], MICMsgLen);
        NdisZeroMemory(EAPOLPKT.KeyDesc.MIC, sizeof(EAPOLPKT.KeyDesc.MIC));

        hmac_md5(pEntry->PTK, LEN_EAP_MICK, (PUCHAR)&EAPOLPKT, (MICMsgLen), mac);
        if (pEntry->WepStatus == Ndis802_11Encryption2Enabled)
        {
            hmac_md5(pEntry->PTK, LEN_EAP_MICK, (PUCHAR)&EAPOLPKT, (MICMsgLen), mac);
        }
        else if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
        {
            UCHAR digest[80];

            HMAC_SHA1((PUCHAR)&EAPOLPKT,  MICMsgLen, pEntry->PTK, LEN_EAP_MICK, digest);
            NdisMoveMemory(mac, digest, LEN_KEY_DESC_MIC);
        }
        if (!NdisEqualMemory(pKeyDesc->MIC, mac, LEN_KEY_DESC_MIC))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("MAC Different in msg 4 of 4-way handshake!\n"));
            break;
        }
        else
            DBGPRINT(RT_DEBUG_TRACE, ("MAC Valid in msg 4 of 4-way handshake!\n"));

        // 3. uses the MLME.SETKEYS.request to configure PTK into MAC
        NdisZeroMemory(&pEntry->PairwiseKey, sizeof(CIPHER_KEY));

        pEntry->PairwiseKey.KeyLen = LEN_TKIP_EK;
        NdisMoveMemory(pEntry->PairwiseKey.Key, &pEntry->PTK[32], LEN_TKIP_EK);
        NdisMoveMemory(pEntry->PairwiseKey.RxMic, &pEntry->PTK[TKIP_AP_RXMICK_OFFSET], LEN_TKIP_RXMICK);
        NdisMoveMemory(pEntry->PairwiseKey.TxMic, &pEntry->PTK[TKIP_AP_TXMICK_OFFSET], LEN_TKIP_TXMICK);
        {
            pEntry->PairwiseKey.CipherAlg = CIPHER_NONE;
            if (pEntry->WepStatus == Ndis802_11Encryption2Enabled)
                pEntry->PairwiseKey.CipherAlg = CIPHER_TKIP;
            else if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
                pEntry->PairwiseKey.CipherAlg = CIPHER_AES;
            AsicAddPairwiseKeyEntry(
                pAd,
                pEntry->Addr,
                (UCHAR)pEntry->Aid, //(pEntry - &pAd->MacTab.Content[0])/sizeof(MAC_TABLE_ENTRY),
                pEntry->PairwiseKey.CipherAlg,
                pEntry->PairwiseKey.Key,
                pEntry->PairwiseKey.TxMic,
                pEntry->PairwiseKey.RxMic);
        }

        // 4. upgrade state
        pEntry->PrivacyFilter = Ndis802_11PrivFilterAcceptAll;
        pEntry->WpaState = AS_PTKINITDONE;
		pEntry->PortSecured = WPA_802_1X_PORT_SECURED;
        pAd->ApCfg.PortSecured = WPA_802_1X_PORT_SECURED;
        ApLogEvent(pAd, pEntry->Addr, EVENT_ASSOCIATED);  // 2005-02-14 log association only after 802.1x successful

        // 5. init Group 2-way handshake if necessary.
        WPAHardTransmit(pAd, pEntry);

	NdisMCancelTimer(&pEntry->RetryTimer, &Cancelled);
        pEntry->ReTryCounter = GROUP_MSG1_RETRY_TIMER_CTR;
        NdisMSetTimer(&pEntry->RetryTimer, PEER_MSG3_RETRY_EXEC_INTV);
    }while(FALSE);

    DBGPRINT(RT_DEBUG_TRACE, ("<=== Wpa1PeerPairMsg4Action\n"));
}

/*
    ==========================================================================
    Description:
        When receiving the last packet of 4-way pairwisekey handshake.
    Return:
    ==========================================================================
*/
VOID Wpa2PeerPairMsg4Action(
    IN PRTMP_ADAPTER    pAd,
    IN MAC_TABLE_ENTRY  *pEntry,
    IN MLME_QUEUE_ELEM  *Elem)
{
    UINT            MicDataLen;
    UCHAR           pDecryp[MAX_LEN_OF_EAP_HS], mic[LEN_KEY_DESC_MIC];
    PUCHAR          pData;
    BOOLEAN         Cancelled;
    AP_KEY_DESCRIPTER  *pKeyDesc;
    VOID            *Msg = &Elem->Msg[LENGTH_802_11];
    UINT            MsgLen = Elem->MsgLen - LENGTH_802_11;
    unsigned long   IrqFlags;

    DBGPRINT(RT_DEBUG_TRACE, ("===>Wpa2PeerPairMsg4Action\n"));

    MicDataLen = MsgLen - LENGTH_802_1_H;
    pData = (PUCHAR)Msg;
    NdisZeroMemory(pDecryp, sizeof(pDecryp));
    NdisZeroMemory(mic, sizeof(mic));

    do
    {
        if ((!pEntry) || (!pEntry->Valid))
            break;

        if (MsgLen < (LENGTH_802_1_H + LENGTH_EAPOL_H + sizeof(AP_KEY_DESCRIPTER) - AP_MAX_LEN_OF_RSNIE - 2))
            break;

        pKeyDesc = (PAP_KEY_DESCRIPTER)(pData + LENGTH_802_1_H + LENGTH_EAPOL_H);

        // 1. verify the Reaply counter, if not valid,
        if (!NdisEqualMemory(pKeyDesc->RCounter, pEntry->R_Counter, LEN_KEY_DESC_REPLAY))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("Replay Counter Different in msg4 of 4-way handshake!\n"));
            DBGPRINT(RT_DEBUG_TRACE, ("Receive : %d %d %d %d \n",pKeyDesc->RCounter[0],pKeyDesc->RCounter[1],pKeyDesc->RCounter[2],pKeyDesc->RCounter[3]));
            DBGPRINT(RT_DEBUG_TRACE, ("Current : %d   %d  %d   %d  \n",pEntry->R_Counter[0],pEntry->R_Counter[1],pEntry->R_Counter[2],pEntry->R_Counter[3]));
            break;
        }
        else
            DBGPRINT(RT_DEBUG_TRACE, ("Replay Counter VALID in msg4 of 4-way handshake!\n"));

        // 2. verify MIC,
        NdisMoveMemory(pDecryp, &pData[LENGTH_802_1_H], MicDataLen);
        NdisZeroMemory(&pDecryp[81], LEN_KEY_DESC_MIC);
        if (pEntry->WepStatus == Ndis802_11Encryption2Enabled)
        {
            hmac_md5(pEntry->PTK,  LEN_EAP_MICK, pDecryp, MicDataLen, mic);
        }
        else
        {
            UCHAR digest[80];

            HMAC_SHA1(pDecryp, MicDataLen, pEntry->PTK, LEN_EAP_MICK, digest);
            NdisMoveMemory(mic, digest, LEN_KEY_DESC_MIC);
        }
        if (!NdisEqualMemory(pKeyDesc->MIC, mic, LEN_KEY_DESC_MIC))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("MIC Different in msg4 of 4-way handshake!\n"));
            break;
        }
        else
            DBGPRINT(RT_DEBUG_TRACE, ("MIC VALID in msg4 of 4-way handshake!\n"));

        NdisZeroMemory(&pEntry->PairwiseKey, sizeof(CIPHER_KEY));

        pEntry->PairwiseKey.KeyLen = LEN_TKIP_EK;
        NdisMoveMemory(pEntry->PairwiseKey.Key, &pEntry->PTK[32], LEN_TKIP_EK);
        NdisMoveMemory(pEntry->PairwiseKey.RxMic, &pEntry->PTK[TKIP_AP_RXMICK_OFFSET], LEN_TKIP_RXMICK);
        NdisMoveMemory(pEntry->PairwiseKey.TxMic, &pEntry->PTK[TKIP_AP_TXMICK_OFFSET], LEN_TKIP_TXMICK);
        {
            pEntry->PairwiseKey.CipherAlg = CIPHER_NONE;
            if (pEntry->WepStatus == Ndis802_11Encryption2Enabled)
                pEntry->PairwiseKey.CipherAlg = CIPHER_TKIP;
            else if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
                pEntry->PairwiseKey.CipherAlg = CIPHER_AES;
            AsicAddPairwiseKeyEntry(
                pAd,
                pEntry->Addr,
                (UCHAR)pEntry->Aid, //(pEntry - &pAd->MacTab.Content[0])/sizeof(MAC_TABLE_ENTRY),
                pEntry->PairwiseKey.CipherAlg,
                pEntry->PairwiseKey.Key,
                pEntry->PairwiseKey.TxMic,
                pEntry->PairwiseKey.RxMic);
        }

        // 3. upgrade state
        pEntry->PrivacyFilter = Ndis802_11PrivFilterAcceptAll;
        pEntry->WpaState = AS_PTKINITDONE;
        pAd->ApCfg.PortSecured = WPA_802_1X_PORT_SECURED;
        ApLogEvent(pAd, pEntry->Addr, EVENT_ASSOCIATED);  // 2005-02-14 log association only after 802.1x successful
        NdisAcquireSpinLock(&pAd->MacTabLock, IrqFlags); // lock to prevent
        NdisMCancelTimer(&pEntry->RetryTimer, &Cancelled);
        NdisReleaseSpinLock(&pAd->MacTabLock, IrqFlags);
        pEntry->GTKState = REKEY_ESTABLISHED;

        if (pAd->ApCfg.GKeyDoneStations > 0 )
            pAd->ApCfg.GKeyDoneStations--;

        if (pAd->ApCfg.GKeyDoneStations == 0)
        {
            pAd->ApCfg.WpaGTKState = SETKEYS_DONE;
        }

        DBGPRINT(RT_DEBUG_TRACE, ("AP SETKEYS DONE - WPA2, AuthMode=%d, WepStatus=%d\n\n", pEntry->AuthMode, pEntry->WepStatus));
    }while(FALSE);
}

/*
    ==========================================================================
    Description:
        When receiving the last packet of 2-way groupkey handshake.
    Return:
    ==========================================================================
*/
VOID PeerGroupMsg2Action(
    IN PRTMP_ADAPTER    pAd,
    IN MAC_TABLE_ENTRY  *pEntry,
    IN VOID             *Msg,
    IN UINT             MsgLen)
{
    UINT            Len;
    UCHAR           pDecryp[MAX_LEN_OF_EAP_HS], mic[LEN_KEY_DESC_MIC], digest[80];
    PUCHAR          pData;
    BOOLEAN         Cancelled;
    AP_KEY_DESCRIPTER  *pKeyDesc;

    do
    {
        if ((!pEntry) || (!pEntry->Valid))
            break;

        if (MsgLen < (LENGTH_802_1_H + LENGTH_EAPOL_H + sizeof(AP_KEY_DESCRIPTER) - AP_MAX_LEN_OF_RSNIE - 2))
            break;

        if (pEntry->WpaState != AS_PTKINITDONE)
            break;

        DBGPRINT(RT_DEBUG_TRACE, ("PeerGroupMsg2Action ====> %x %x %x %x %x %x \n",pEntry->Addr[0], pEntry->Addr[1],
            pEntry->Addr[2],pEntry->Addr[3],pEntry->Addr[4],pEntry->Addr[5]));

        Len = MsgLen - LENGTH_802_1_H;
        pData = (PUCHAR)Msg;
        NdisZeroMemory(pDecryp, sizeof(pDecryp));
        NdisZeroMemory(mic, sizeof(mic));

        pKeyDesc = (PAP_KEY_DESCRIPTER)(pData + LENGTH_802_1_H + LENGTH_EAPOL_H);

        // 1. verify the Reaply counter, if not valid,
        if (!NdisEqualMemory(pKeyDesc->RCounter, pEntry->R_Counter, LEN_KEY_DESC_REPLAY))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("Replay Counter  Different  in msg 2 of GROUP 2-way handshake! \n"));
            DBGPRINT(RT_DEBUG_ERROR, ("Receive: %d %d %d %d - %d %d %d %d \n",pKeyDesc->RCounter[0],pKeyDesc->RCounter[1],pKeyDesc->RCounter[2],pKeyDesc->RCounter[3],pKeyDesc->RCounter[4],pKeyDesc->RCounter[5],pKeyDesc->RCounter[6],pKeyDesc->RCounter[7]));
            DBGPRINT(RT_DEBUG_ERROR, ("Current: %d %d %d %d - %d %d %d %d \n",pEntry->R_Counter[0],pEntry->R_Counter[1],pEntry->R_Counter[2],pEntry->R_Counter[3],pEntry->R_Counter[4],pEntry->R_Counter[5],pEntry->R_Counter[6],pEntry->R_Counter[7]));
            break;
        }
        else
            DBGPRINT(RT_DEBUG_TRACE, ("Replay Counter VALID in msg 2 of GROUP 2-way handshake! \n"));

        // 2. verify MIC,
        NdisMoveMemory(pDecryp, &pData[LENGTH_802_1_H], Len);
        NdisZeroMemory(&pDecryp[81], LEN_KEY_DESC_MIC);
        if (pEntry->WepStatus == Ndis802_11Encryption2Enabled)
        {
            hmac_md5(pEntry->PTK,  LEN_EAP_MICK, pDecryp, Len, mic);
        }
        else
        {
            //encrypt_mpdu(PTK,  pnl,  pnh, 0, PAY_LEN, &LEN_HDR, (INT*)&MICMsgLen, 0, 0, OutBuffer, mac);
            HMAC_SHA1(pDecryp,  Len, pEntry->PTK, LEN_EAP_MICK, digest);
            NdisMoveMemory(mic, digest, LEN_KEY_DESC_MIC);
        }
        if (!NdisEqualMemory(pKeyDesc->MIC, mic, LEN_KEY_DESC_MIC))
        {
            DBGPRINT(RT_DEBUG_ERROR, ("MIC Different in msg 2 of 2-way handshake!\n"));
            break;
        }
        else
            DBGPRINT(RT_DEBUG_TRACE, ("MIC VALID in msg 2 of 2-way handshake!\n"));

        // 3.  upgrade state
        NdisMCancelTimer(&pEntry->RetryTimer, &Cancelled);

        pEntry->GTKState = REKEY_ESTABLISHED;

        if (pAd->ApCfg.GKeyDoneStations > 0 )
            pAd->ApCfg.GKeyDoneStations--;

        if (pAd->ApCfg.GKeyDoneStations == 0)
        {
            pAd->ApCfg.WpaGTKState = SETKEYS_DONE;
        }

        DBGPRINT(RT_DEBUG_TRACE, ("AP SETKEYS DONE - WPA1, AuthMode=%d, WepStatus=%d\n\n", pEntry->AuthMode, pEntry->WepStatus));
    }while(FALSE);
}

/*
    ==========================================================================
    Description:
        Only for sending the first packet of 2-way groupkey handshake
    Return:
    ==========================================================================
*/
NDIS_STATUS WPAHardTransmit(
    IN PRTMP_ADAPTER    pAd,
    IN MAC_TABLE_ENTRY  *pEntry)
{
    UCHAR               *GTK, *pRc4GTK, *mic, *digest;
    UCHAR               *Key_Data;
    UCHAR               ekey[LEN_KEY_DESC_IV + LEN_EAP_EK];
    ULONG               FrameLen = 0;
    PUCHAR              pOutBuffer = NULL;
    UCHAR               Header802_3[14];
    AP_EAPOL_PACKET     Packet;
    UINT                i;
    NDIS_STATUS         NStatus;
	UCHAR				*mpool = kmalloc(TKIP_GTK_LENGTH + TKIP_GTK_LENGTH + LEN_KEY_DESC_MIC + 80 + 512 + 32, GFP_ATOMIC);

	if(mpool == NULL)
		return NDIS_STATUS_SUCCESS;

	/* GTK  Len = TKIP_GTK_LENGTH */
	GTK = (UCHAR *) ROUND_UP(mpool, 4);
	/* pRc4GTK Len = TKIP_GTK_LENGTH */
	pRc4GTK = (UCHAR *) ROUND_UP(GTK + TKIP_GTK_LENGTH, 4);
	/* mic Len = LEN_KEY_DESC_MIC */
	mic = (UCHAR *) ROUND_UP(pRc4GTK + TKIP_GTK_LENGTH, 4);
	/* digest Len = 80 */
	digest = (UCHAR *) ROUND_UP(mic + LEN_KEY_DESC_MIC, 4);
	/* Key_Data Len = 512 */
	Key_Data = (UCHAR *) ROUND_UP(digest + 80, 4);

    NdisZeroMemory(ekey, sizeof(ekey));
    NdisZeroMemory(pRc4GTK, sizeof(pRc4GTK));

    do
    {
        if ((!pEntry) || (!pEntry->Valid))
            break;

        NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);
        if (NStatus != NDIS_STATUS_SUCCESS)
            break;

        // Increment replay counter by 1
        i = LEN_KEY_DESC_REPLAY;
        do
        {
            i--;
            pEntry->R_Counter[i]++;
            if (i == 0)
            {
                break;
            }
        }
        while (pEntry->R_Counter[i] == 0);

        // 0. init Packet and Fill header
        NdisZeroMemory(&Packet, sizeof(Packet));
        Packet.ProVer = EAPOL_VER;
        Packet.ProType = EAPOLKey;
        Packet.Body_Len[1] = LEN_MSG1_2WAY;

        // 2, Fill key version, keyinfo, key len
        if ((pEntry->AuthMode == Ndis802_11AuthModeWPA2) ||
            (pEntry->AuthMode == Ndis802_11AuthModeWPA2PSK))
        {
            Packet.KeyDesc.Type = WPA2_KEY_DESC;
            Packet.KeyDesc.Keyinfo.EKD_DL = 1;
        }
        else
            Packet.KeyDesc.Type = WPA1_KEY_DESC;

        Packet.KeyDesc.Keyinfo.KeyMic = 1;
        Packet.KeyDesc.Keyinfo.Secure = 1;
        Packet.KeyDesc.Keyinfo.KeyDescVer = DESC_TYPE_TKIP;
        Packet.KeyDesc.Keyinfo.KeyType = GROUPKEY;
        Packet.KeyDesc.Keyinfo.KeyIndex = pAd->CommonCfg.DefaultKeyId;
        Packet.KeyDesc.Keyinfo.KeyAck = 1;
        Packet.KeyDesc.KeyLength[1] = TKIP_GTK_LENGTH;
        Packet.KeyDesc.DataLen[1] = TKIP_GTK_LENGTH;
        if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
        {
            Packet.Body_Len[1] -= 8;
            Packet.KeyDesc.KeyLength[1] = LEN_AES_KEY;
            Packet.KeyDesc.DataLen[1] = LEN_AES_KEY + 8;
            Packet.KeyDesc.Keyinfo.KeyDescVer = DESC_TYPE_AES;
        }

        CountGTK(pAd->ApCfg.GMK, (UCHAR*)pAd->ApCfg.GNonce, pAd->CurrentAddress, GTK, TKIP_GTK_LENGTH);
        pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].KeyLen = LEN_TKIP_EK;
        NdisMoveMemory(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].Key, GTK, LEN_TKIP_EK);
        NdisMoveMemory(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxMic, &GTK[16], LEN_TKIP_TXMICK);
        NdisMoveMemory(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].RxMic, &GTK[24], LEN_TKIP_RXMICK);

        if (pEntry->WepStatus == Ndis802_11Encryption2Enabled)
            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg = CIPHER_TKIP;
        else if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg = CIPHER_AES;
        else
            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg = CIPHER_NONE;

        // install Group Key to MAC ASIC
//        RTUSBEnqueueInternalCmd(pAd, RT_OID_ASIC_ADD_SKEY, NULL, 0);
        /* ~~sample, 2006/10/03, modification, use RTUSBEnqueueCmdFromNdis not RTUSBEnqueueInternalCmd,
           because no RT_OID_ASIC_ADD_SKEY support in RTUSBEnqueueInternalCmd() */
        RTUSBEnqueueCmdFromNdis(pAd, RT_OID_ASIC_ADD_SKEY, FALSE, NULL, 0);
       /* AsicAddSharedKeyEntry(
            pAd,
            BSS0,
            pAd->CommonCfg.DefaultKeyId,
            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].CipherAlg,
            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].Key,
            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxMic,
            pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].RxMic);*/

        if ((pEntry->AuthMode == Ndis802_11AuthModeWPA2) ||
            (pEntry->AuthMode == Ndis802_11AuthModeWPA2PSK))
        {
            UCHAR  key_length ;

            NdisZeroMemory(Key_Data, 512); //sizeof(Key_Data));
            NdisZeroMemory(Packet.KeyDesc.MIC, sizeof(Packet.KeyDesc.MIC));

            Packet.Body_Len[1] = 0;
            // Key Data Encapsulation format
            Key_Data[0] = 0xDD;
            Key_Data[2] = 0x00;
            Key_Data[3] = 0x0F;
            Key_Data[4] = 0xAC;
            Key_Data[5] = 0x01;

            // GTK Key Data Encapsulation format
            Key_Data[6] = (pAd->CommonCfg.DefaultKeyId & 0x03);
            Key_Data[7] = 0x00;

            // handle the difference between TKIP and AES-CCMP
            if (pAd->CommonCfg.GroupKeyWepStatus == Ndis802_11Encryption3Enabled)
            {
                Key_Data[1] = 0x16;// 4+2+16(OUI+GTK+GTKKEY)
                NdisMoveMemory(&Key_Data[8], GTK, LEN_AES_KEY);
                Packet.Body_Len[1] += 8+LEN_AES_KEY;
            }
            else
            {
                Key_Data[1] = 0x26;// 4+2+32(OUI+GTK+GTKKEY)
                NdisMoveMemory(&Key_Data[8], GTK, TKIP_GTK_LENGTH);
                Packet.Body_Len[1] += 8+TKIP_GTK_LENGTH;
            }

            // Patch for compatibility between zero config and funk
            if (pAd->CommonCfg.GroupKeyWepStatus == Ndis802_11Encryption3Enabled)
            {
                Key_Data[8+TKIP_GTK_LENGTH] = 0xDD;
                Key_Data[8+TKIP_GTK_LENGTH+1] = 0;
                Packet.Body_Len[1] += 2;
            }
            else
            {
                Key_Data[8+TKIP_GTK_LENGTH] = 0xDD;
                Key_Data[8+TKIP_GTK_LENGTH+1] = 0;
                Key_Data[8+TKIP_GTK_LENGTH+2] = 0;
                Key_Data[8+TKIP_GTK_LENGTH+3] = 0;
                Key_Data[8+TKIP_GTK_LENGTH+4] = 0;
                Key_Data[8+TKIP_GTK_LENGTH+5] = 0;
                Packet.Body_Len[1] += 6;
            }

            key_length = Packet.Body_Len[1];
            Packet.KeyDesc.DataLen[1] = key_length;
            Packet.Body_Len[1] +=93 +2;

            NdisZeroMemory(mic,LEN_KEY_DESC_MIC );

            NdisMoveMemory(Packet.KeyDesc.RSC, pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxTsc, 6);

            NdisMoveMemory(Packet.KeyDesc.RCounter, pEntry->R_Counter, LEN_KEY_DESC_REPLAY);
            NdisZeroMemory(Packet.KeyDesc.Nonce, LEN_KEY_DESC_NONCE);//WPA2 0;
            NdisZeroMemory(Packet.KeyDesc.IV,sizeof(LEN_KEY_DESC_IV));//WPA2 0; WPA1 random
            // count MIC
            if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
            {
                AES_GTK_KEY_WRAP(&pEntry->PTK[16], Key_Data, key_length, pRc4GTK);
                // AES wrap function will grow 8 bytes in length
                NdisMoveMemory(Packet.KeyDesc.Data, pRc4GTK, (key_length + 8));
                Packet.Body_Len[1] += 8;
                Packet.KeyDesc.DataLen[1] += 8;
                MakeOutgoingFrame(pOutBuffer,           &FrameLen,
                                Packet.Body_Len[1] + 4, &Packet,
                               END_OF_ARGS);

                HMAC_SHA1(pOutBuffer, FrameLen, pEntry->PTK, LEN_EAP_MICK, digest);
                NdisMoveMemory(Packet.KeyDesc.MIC, digest, LEN_KEY_DESC_MIC);
            }
            else
            {
                // PREPARE Encrypted  "Key DATA" field.  (Encrypt GTK with RC4, usinf PTK[16]->[31] as Key, IV-field as IV)
                // put TxTsc in Key RSC field
                pAd->PrivateInfo.FCSCRC32 = PPPINITFCS32;   //Init crc32.

                // ekey is the contanetion of IV-field, and PTK[16]->PTK[31]
                NdisMoveMemory(ekey, Packet.KeyDesc.IV, LEN_KEY_DESC_IV);
                NdisMoveMemory(&ekey[LEN_KEY_DESC_IV], &pEntry->PTK[16], LEN_EAP_EK);
                ARCFOUR_INIT(&pAd->PrivateInfo.WEPCONTEXT, ekey, (LEN_KEY_DESC_IV+LEN_EAP_EK));  //INIT SBOX, KEYLEN+3(IV)
                pAd->PrivateInfo.FCSCRC32 = RTMP_CALC_FCS32(pAd->PrivateInfo.FCSCRC32, Key_Data, key_length);
                WPAARCFOUR_ENCRYPT(&pAd->PrivateInfo.WEPCONTEXT, pRc4GTK,Key_Data, key_length);
                NdisMoveMemory(Packet.KeyDesc.Data, pRc4GTK, key_length);

                // make a frame for Countint MIC,
                MakeOutgoingFrame(pOutBuffer,           &FrameLen,
                                    Packet.Body_Len[1] + 4, &Packet,
                                    END_OF_ARGS);

                hmac_md5(pEntry->PTK, LEN_EAP_MICK, pOutBuffer, FrameLen, mic);
                NdisMoveMemory(Packet.KeyDesc.MIC, mic, LEN_KEY_DESC_MIC);
            }
//            WPAMake8023Hdr(pAd, pEntry, Header802_3);
//            MakeOutgoingFrame(pOutBuffer,            &FrameLen,
//                            LENGTH_802_3,            Header802_3,
//                            Packet.Body_Len[1] + 4,  &Packet,
//                            END_OF_ARGS);

//            DBGPRINT(RT_DEBUG_TRACE, "RTMPToWirelessSta : ETHTYPE = %x %x FrameLen = %d! \n",Header802_3[12],Header802_3[13],FrameLen);

            MAKE_802_3_HEADER(Header802_3, pEntry->Addr, pAd->CurrentAddress, EAPOL);
            APToWirelessSta(pAd, pEntry->Aid, Header802_3, LENGTH_802_3, (PUCHAR) &Packet, Packet.Body_Len[1] + 4);
//            RTMPToWirelessSta(pAdapter, OutBuffer, FrameLen);

        }  // WPA2 group key
        else
        {// WPA1 group key

            MAKE_802_3_HEADER(Header802_3, pEntry->Addr, pAd->CurrentAddress, EAPOL);

            NdisMoveMemory(Packet.KeyDesc.RCounter, pEntry->R_Counter, LEN_KEY_DESC_REPLAY);
            NdisMoveMemory(Packet.KeyDesc.Nonce, pAd->ApCfg.GNonce, LEN_KEY_DESC_NONCE);
            NdisMoveMemory(Packet.KeyDesc.IV, &pAd->ApCfg.GNonce[16], sizeof(LEN_KEY_DESC_IV));

            // Suggest IV be random number plus some number,
            Packet.KeyDesc.IV[15] += 2;
            NdisMoveMemory(Packet.KeyDesc.RSC, pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxTsc, 6);
            // Count EAPOL MIC , and encrypt DATA field before Send,   DATA fields includes the encrypted GTK
            if (pEntry->WepStatus == Ndis802_11Encryption3Enabled)
            {
                if (pAd->CommonCfg.GroupKeyWepStatus == Ndis802_11Encryption3Enabled)
                {
                    AES_GTK_KEY_WRAP(&pEntry->PTK[16], GTK, 16, pRc4GTK);
                    NdisMoveMemory(Packet.KeyDesc.Data, pRc4GTK, (LEN_AES_KEY +8));
                }
                else
                {
                    AES_GTK_KEY_WRAP(&pEntry->PTK[16], GTK, TKIP_GTK_LENGTH, pRc4GTK);
                    NdisMoveMemory(Packet.KeyDesc.Data, pRc4GTK, (TKIP_GTK_LENGTH +8));
                    Packet.KeyDesc.KeyLength[1] += 16;
                    Packet.KeyDesc.DataLen[1] += 16;
                    Packet.Body_Len[1] += 16;
                }

                // First make a frame  for Countint MIC,
                MakeOutgoingFrame(pOutBuffer,               &FrameLen,
                                  Packet.Body_Len[1] + 4,   &Packet,
                                  END_OF_ARGS);

                HMAC_SHA1(pOutBuffer,  FrameLen, pEntry->PTK, LEN_EAP_MICK, digest);
                NdisMoveMemory(&Packet.KeyDesc.MIC, digest, LEN_KEY_DESC_MIC);
            }
            else
            {
                // PREPARE Encrypted  "Key DATA" field.  (Encrypt GTK with RC4, usinf PTK[16]->[31] as Key, IV-field as IV)
                // put TxTsc in Key RSC field
                pAd->PrivateInfo.FCSCRC32 = PPPINITFCS32;   //Init crc32.

                // ekey is the contanetion of IV-field, and PTK[16]->PTK[31]
                NdisMoveMemory(ekey, Packet.KeyDesc.IV, LEN_KEY_DESC_IV);
                NdisMoveMemory(&ekey[LEN_KEY_DESC_IV], &pEntry->PTK[16], LEN_EAP_EK);
                ARCFOUR_INIT(&pAd->PrivateInfo.WEPCONTEXT, ekey, sizeof(ekey));  //INIT SBOX, KEYLEN+3(IV)
                pAd->PrivateInfo.FCSCRC32 = RTMP_CALC_FCS32(pAd->PrivateInfo.FCSCRC32, GTK, TKIP_GTK_LENGTH);
                WPAARCFOUR_ENCRYPT(&pAd->PrivateInfo.WEPCONTEXT, pRc4GTK,GTK,  TKIP_GTK_LENGTH);
                NdisMoveMemory(Packet.KeyDesc.Data,  pRc4GTK, TKIP_GTK_LENGTH);

                // make a frame for Countint MIC,
                MakeOutgoingFrame(pOutBuffer,               &FrameLen,
                                  Packet.Body_Len[1] + 4,   &Packet,
                                  END_OF_ARGS);

                hmac_md5(pEntry->PTK, LEN_EAP_MICK, pOutBuffer, FrameLen, mic);
                NdisMoveMemory(Packet.KeyDesc.MIC, mic, LEN_KEY_DESC_MIC);
            }

            DBGPRINT(RT_DEBUG_TRACE, ("WPAHardTransmit : FrameLen = %d \n", Packet.Body_Len[1] + 4));
            APToWirelessSta(pAd, pEntry->Aid, Header802_3, LENGTH_802_3, (PUCHAR) &Packet, Packet.Body_Len[1] + 4);
        }//WPA1 group key
    }while (FALSE);

    MlmeFreeMemory(pAd, pOutBuffer);

	kfree(mpool);
    return (NDIS_STATUS_SUCCESS);
}

VOID DisAssocAction(
    IN PRTMP_ADAPTER    pAd,
    IN MAC_TABLE_ENTRY  *pEntry,
    IN USHORT           Reason)
{
    PUCHAR          pOutBuffer = NULL;
    ULONG           FrameLen = 0;
    HEADER_802_11   DisassocHdr;
    NDIS_STATUS     NStatus;

    if (pEntry)
    {
        //  send out a DISASSOC request frame
        NStatus = MlmeAllocateMemory(pAd, (PVOID)&pOutBuffer);
        if (NStatus != NDIS_STATUS_SUCCESS)
            return;

        DBGPRINT(RT_DEBUG_TRACE, ("ASSOC - Send DISASSOC Reason = %d frame to %x %x %x %x %x %x \n",Reason,pEntry->Addr[0],
            pEntry->Addr[1],pEntry->Addr[2],pEntry->Addr[3],pEntry->Addr[4],pEntry->Addr[5]));

        MgtMacHeaderInit(pAd, &DisassocHdr, SUBTYPE_DISASSOC, 0, pEntry->Addr, pAd->CommonCfg.Bssid);
        MakeOutgoingFrame(pOutBuffer,               &FrameLen,
                          sizeof(HEADER_802_11),    &DisassocHdr,
                          2,                        &Reason,
                          END_OF_ARGS);
        MiniportMMRequest(pAd, pOutBuffer, FrameLen);


        // ApLogEvent(pAd, pEntry->Addr, EVENT_DISASSOCIATED);
        MacTableDeleteEntry(pAd, pEntry->Addr);
    }
}

/*
    ==========================================================================
    Description:
        Timer execution function for periodically updating group key.
    Return:
    ==========================================================================
*/
VOID GREKEYPeriodicExec(
    IN PVOID SystemSpecific1,
    IN PVOID FunctionContext,
    IN PVOID SystemSpecific2,
    IN PVOID SystemSpecific3)
{
    UINT            i;
    ULONG           temp_counter = 0;
    BOOLEAN         Cancelled;
    PRTMP_ADAPTER   pAd = (PRTMP_ADAPTER)FunctionContext;

    DBGPRINT(RT_DEBUG_INFO, ("GROUP REKEY PeriodicExec ==>> \n"));

    /* ~~sample, modification for WPA/WPA2 PSK rekey */
//    if((pAd->CommonCfg.AuthMode < Ndis802_11AuthModeWPA) ||  (pAd->CommonCfg.AuthMode == Ndis802_11AuthModeWPA2PSK)
//		            ||  (pAd->CommonCfg.AuthMode ==  Ndis802_11AuthModeWPA1PSKWPA2PSK))
    if ((pAd->CommonCfg.AuthMode != Ndis802_11AuthModeWPAPSK) &&
        (pAd->CommonCfg.AuthMode != Ndis802_11AuthModeWPA2PSK) &&
        (pAd->CommonCfg.AuthMode != Ndis802_11AuthModeWPA1PSKWPA2PSK))
    {
		NdisMCancelTimer(&pAd->ApCfg.REKEYTimer, &Cancelled);
		pAd->ApCfg.REKEYTimerRunning = FALSE;
		pAd->ApCfg.REKEYTimer.Repeat = FALSE;
		return;
    }

    if ((pAd->ApCfg.WPAREKEY.ReKeyMethod == TIME_REKEY) && (pAd->ApCfg.REKEYCOUNTER < 0xffffffff))
        temp_counter = (++pAd->ApCfg.REKEYCOUNTER);
    // REKEYCOUNTER is incremented every TX_RING_SIZE packets transmitted,
    // But the unit of Rekeyinterval is 1K packets
    else if (pAd->ApCfg.WPAREKEY.ReKeyMethod == PKT_REKEY )
//        temp_counter = pAd->ApCfg.REKEYCOUNTER/TX_RING_SIZE;
        /* ~~sample, 2006/10/04, PKT_REKEY function is not work so I modify the function */
        /* REKEYCOUNTER will be incremented when a packet is tx */
        temp_counter = pAd->ApCfg.REKEYCOUNTER >> 10; /* REKEYCOUNTER/1024 */
    else
    {
//        NdisMCancelTimer(&pAd->ApCfg.REKEYTimer, &Cancelled);
        pAd->ApCfg.REKEYTimerRunning = FALSE;
        pAd->ApCfg.REKEYTimer.Repeat = FALSE;
        return;
    }

//    if (temp_counter > (pAd->ApCfg.WPAREKEY.ReKeyInterval))
    if (temp_counter >= (pAd->ApCfg.WPAREKEY.ReKeyInterval)) /* ~~sample, 2006/10/4, fix */
    {
        pAd->ApCfg.REKEYCOUNTER = 0;
        DBGPRINT(RT_DEBUG_TRACE, ("Rekey Interval Excess, GKeyDoneStations=%d\n", pAd->MacTab.Size));
        pAd->ApCfg.WpaGTKState = SETKEYS;

        // take turn updating different groupkey index,
        if ((pAd->ApCfg.GKeyDoneStations = pAd->MacTab.Size) > 0)
        {
            /* ~~sample, bug fix, only use keyIdx = 1 in window version,
               so can not swap DefaultKeyId here, this is a window bug */
//            pAd->CommonCfg.DefaultKeyId = (pAd->CommonCfg.DefaultKeyId == 1) ? 2 : 1;
            APGenRandom(pAd, pAd->ApCfg.GNonce);
            for (i = 0; i < MAX_LEN_OF_MAC_TABLE; i++)
            {
                if ((pAd->MacTab.Content[i].Valid == TRUE) && (pAd->MacTab.Content[i].WpaState == AS_PTKINITDONE))
                {
                    pAd->MacTab.Content[i].GTKState = REKEY_NEGOTIATING;
                    NdisZeroMemory(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxTsc, sizeof(pAd->SharedKey[BSS0][pAd->CommonCfg.DefaultKeyId].TxTsc));
                    WPAHardTransmit(pAd, &pAd->MacTab.Content[i]);
                    DBGPRINT(RT_DEBUG_TRACE, ("Rekey interval excess, Update Group Key for  %x %x %x  %x %x %x , DefaultKeyId= %x \n",\
                        pAd->MacTab.Content[i].Addr[0],pAd->MacTab.Content[i].Addr[1],\
                        pAd->MacTab.Content[i].Addr[2],pAd->MacTab.Content[i].Addr[3],\
                        pAd->MacTab.Content[i].Addr[4],pAd->MacTab.Content[i].Addr[5],\
                        pAd->CommonCfg.DefaultKeyId));
                }
            }
        }
    }
}

VOID WPAMacHeaderInit(
    IN PRTMP_ADAPTER        pAd,
    IN OUT PHEADER_802_11   Hdr,
    IN UCHAR                wep,
    IN PUCHAR               pAddr1)
{
    NdisZeroMemory(Hdr, sizeof(HEADER_802_11));
    Hdr->FC.Type = BTYPE_DATA;
    Hdr->FC.FrDs = 1;
    if (wep == 1)
        Hdr->FC.Wep = 1;

     // Addr1: DA, Addr2: BSSID, Addr3: SA
    COPY_MAC_ADDR(Hdr->Addr1, pAddr1);
    COPY_MAC_ADDR(Hdr->Addr2, pAd->CommonCfg.Bssid);
    COPY_MAC_ADDR(Hdr->Addr3, pAd->CurrentAddress);
    Hdr->Sequence = pAd->Sequence;
}

VOID CountPTK(
    IN  UCHAR   *PMK,
    IN  UCHAR   *ANonce,
    IN  UCHAR   *AA,
    IN  UCHAR   *SNonce,
    IN  UCHAR   *SA,
    OUT UCHAR   *output,
    IN  UINT    len)
{
    UCHAR   concatenation[76];
    UINT    CurrPos=0;
    UCHAR   Prefix[22];
    UCHAR       temp[32];

    NdisZeroMemory(temp, sizeof(temp));

    GetSmall(SA, AA, temp, 6);
    NdisMoveMemory(concatenation, temp, 6);
    CurrPos += 6;

    GetLarge(SA, AA, temp, 6);
    NdisMoveMemory(&concatenation[CurrPos], temp, 6);
    CurrPos += 6;

    GetSmall(ANonce, SNonce, temp, 32);
    NdisMoveMemory(&concatenation[CurrPos], temp, 32);
    CurrPos += 32;

    GetLarge(ANonce, SNonce, temp, 32);
    NdisMoveMemory(&concatenation[CurrPos], temp, 32);
    CurrPos += 32;

    Prefix[0] = 'P';
    Prefix[1] = 'a';
    Prefix[2] = 'i';
    Prefix[3] = 'r';
    Prefix[4] = 'w';
    Prefix[5] = 'i';
    Prefix[6] = 's';
    Prefix[7] = 'e';
    Prefix[8] = ' ';
    Prefix[9] = 'k';
    Prefix[10] = 'e';
    Prefix[11] = 'y';
    Prefix[12] = ' ';
    Prefix[13] = 'e';
    Prefix[14] = 'x';
    Prefix[15] = 'p';
    Prefix[16] = 'a';
    Prefix[17] = 'n';
    Prefix[18] = 's';
    Prefix[19] = 'i';
    Prefix[20] = 'o';
    Prefix[21] = 'n';
    PRF(PMK, PMK_LEN, Prefix,  22, concatenation, 76 , output, len);
}

VOID CountGTK(
    IN  UCHAR   *GMK,
    IN  UCHAR   *GNonce,
    IN  UCHAR   *AA,
    OUT UCHAR   *output,
    IN  UINT    len)
{
    UCHAR   concatenation[76];
    UINT    CurrPos=0;
    UCHAR   Prefix[19];
    UCHAR   temp[80];

    NdisMoveMemory(&concatenation[CurrPos], AA, 6);
    CurrPos += 6;

    NdisMoveMemory(&concatenation[CurrPos], GNonce , 32);
    CurrPos += 32;

    Prefix[0] = 'G';
    Prefix[1] = 'r';
    Prefix[2] = 'o';
    Prefix[3] = 'u';
    Prefix[4] = 'p';
    Prefix[5] = ' ';
    Prefix[6] = 'k';
    Prefix[7] = 'e';
    Prefix[8] = 'y';
    Prefix[9] = ' ';
    Prefix[10] = 'e';
    Prefix[11] = 'x';
    Prefix[12] = 'p';
    Prefix[13] = 'a';
    Prefix[14] = 'n';
    Prefix[15] = 's';
    Prefix[16] = 'i';
    Prefix[17] = 'o';
    Prefix[18] = 'n';

    PRF(GMK, PMK_LEN, Prefix,  19, concatenation, 38 , temp, len);
    NdisMoveMemory(output, temp, len);
}

VOID GetSmall(
    IN  PVOID   pSrc1,
    IN  PVOID   pSrc2,
    OUT PUCHAR  pOut,
    IN  ULONG   Length)
{
    PUCHAR  pMem1;
    PUCHAR  pMem2;
    ULONG   Index = 0;

    pMem1 = (PUCHAR) pSrc1;
    pMem2 = (PUCHAR) pSrc2;

    for (Index = 0; Index < Length; Index++)
    {
        if (pMem1[Index] != pMem2[Index])
        {
            if (pMem1[Index] > pMem2[Index])
                NdisMoveMemory(pOut, pSrc2, Length);
            else
                NdisMoveMemory(pOut, pSrc1, Length);

            break;
        }
    }
}

VOID GetLarge(
    IN  PVOID   pSrc1,
    IN  PVOID   pSrc2,
    OUT PUCHAR  pOut,
    IN  ULONG   Length)
{
    PUCHAR  pMem1;
    PUCHAR  pMem2;
    ULONG   Index = 0;

    pMem1 = (PUCHAR) pSrc1;
    pMem2 = (PUCHAR) pSrc2;

    for (Index = 0; Index < Length; Index++)
    {
        if (pMem1[Index] != pMem2[Index])
        {
            if (pMem1[Index] > pMem2[Index])
                NdisMoveMemory(pOut, pSrc1, Length);
            else
                NdisMoveMemory(pOut, pSrc2, Length);

            break;
        }
    }
}

// 802.1i  Annex F.9
VOID APGenRandom(
    IN PRTMP_ADAPTER pAd,
    OUT UCHAR       *random)
{
    INT             i, curr;
    UCHAR           local[80];
    UCHAR           result[80];
    LARGE_INTEGER   CurrentTime;
	UCHAR	        prefix[] = {'I', 'n', 'i', 't', ' ', 'C', 'o', 'u', 'n', 't', 'e', 'r'};

    NdisZeroMemory(result, 80);
    NdisZeroMemory(local, 80);
    NdisMoveMemory(local, pAd->CurrentAddress, MAC_ADDR_LEN);

    for (i = 0; i < 32; i++)
    {
        curr =  MAC_ADDR_LEN;
        NdisGetCurrentSystemTime(&CurrentTime);
        NdisMoveMemory(local,  pAd->CurrentAddress, MAC_ADDR_LEN);
        curr += MAC_ADDR_LEN;
        NdisMoveMemory(&local[curr],  &CurrentTime, sizeof(CurrentTime));
        curr += sizeof(CurrentTime);
        NdisMoveMemory(&local[curr],  result, 32);
        curr += 32;
        NdisMoveMemory(&local[curr],  &i,  2);
        curr += 2;
        PRF(pAd->ApCfg.Key_Counter, 32, prefix,12, local, curr, result, 32);
    }

    for (i = 32; i > 0; i--)
    {
        if (pAd->ApCfg.Key_Counter[i-1] == 0xff)
        {
            pAd->ApCfg.Key_Counter[i-1] = 0;
        }
        else
        {
            pAd->ApCfg.Key_Counter[i-1]++;
            break;
        }
    }
    NdisMoveMemory(random, result,  32);
}

/*
    ==========================================================================
    Description:
        ENCRYPT AES GTK before sending in EAPOL frame.
        AES GTK length = 128 bit,  so fix blocks for aes-key-wrap as 2 in this function.
        This function references to RFC 3394 for aes key wrap algorithm.
    Return:
    ==========================================================================
*/
VOID AES_GTK_KEY_WRAP(
    IN UCHAR    *key,
    IN UCHAR    *plaintext,
    IN UCHAR    p_len,
    OUT UCHAR   *ciphertext)
{
    UCHAR       A[8], BIN[16], BOUT[16];
    UCHAR       *R = kmalloc(512, GFP_ATOMIC);
    INT         num_blocks = p_len/8;   // unit:64bits
    INT         i, j;
    aes_context aesctx;
    UCHAR       xor;

	if(R == NULL)
		return;

    aes_set_key(&aesctx, key, 128);

    // Init IA
    for (i = 0; i < 8; i++)
        A[i] = 0xa6;

    //Input plaintext
    for (i = 0; i < num_blocks; i++)
    {
        for (j = 0 ; j < 8; j++)
            R[8 * (i + 1) + j] = plaintext[8 * i + j];
    }

    // Key Mix
    for (j = 0; j < 6; j++)
    {
        for(i = 1; i <= num_blocks; i++)
        {
            //phase 1
            memcpy(BIN, A, 8);
            memcpy(&BIN[8], &R[8 * i], 8);
            aes_encrypt(&aesctx, BIN, BOUT);

            memcpy(A, &BOUT[0], 8);
            xor = num_blocks * j + i;
            A[7] = BOUT[7] ^ xor;
            memcpy(&R[8 * i], &BOUT[8], 8);
        }
    }

    // Output ciphertext
    memcpy(ciphertext, A, 8);

    for (i = 1; i <= num_blocks; i++)
    {
        for (j = 0 ; j < 8; j++)
            ciphertext[8 * i + j] = R[8 * i + j];
    }

	kfree(R);
	return;
}

VOID RTMPMakeRSNIE(
    IN  PRTMP_ADAPTER   pAd,
    IN  UINT            AuthMode,
    IN  UINT            WepStatus)
{
	if (WepStatus == Ndis802_11Encryption4Enabled)
        pAd->CommonCfg.GroupKeyWepStatus = Ndis802_11Encryption2Enabled;
    else
        pAd->CommonCfg.GroupKeyWepStatus = WepStatus;

    if ((AuthMode != Ndis802_11AuthModeWPA) && (AuthMode != Ndis802_11AuthModeWPAPSK)
        && (AuthMode != Ndis802_11AuthModeWPA2) && (AuthMode != Ndis802_11AuthModeWPA2PSK)
        && (AuthMode != Ndis802_11AuthModeWPA1WPA2) && (AuthMode != Ndis802_11AuthModeWPA1PSKWPA2PSK))
        return;

    pAd->ApCfg.RSNIE_Len[0] = 0;
    pAd->ApCfg.RSNIE_Len[1] = 0;
    NdisZeroMemory(pAd->ApCfg.RSN_IE[0], AP_MAX_LEN_OF_RSNIE);
    NdisZeroMemory(pAd->ApCfg.RSN_IE[1], AP_MAX_LEN_OF_RSNIE);

    // For WPA1, RSN_IE=221
    if ((AuthMode == Ndis802_11AuthModeWPA) || (AuthMode == Ndis802_11AuthModeWPAPSK)
        || (AuthMode == Ndis802_11AuthModeWPA1WPA2) || (AuthMode == Ndis802_11AuthModeWPA1PSKWPA2PSK))
    {
        RSNIE               *pRsnie;
        RSNIE_AUTH          *pRsnie_auth;
        RSN_CAPABILITIES    *pRSN_Cap;
        UCHAR               Rsnie_size = 0;

        pRsnie = (RSNIE*)pAd->ApCfg.RSN_IE[0];
        NdisMoveMemory(pRsnie->oui, OUI_WPA_WEP40, 4);
        pRsnie->version = 1;

        switch (WepStatus)
        {
            case Ndis802_11Encryption2Enabled:
                NdisMoveMemory(pRsnie->mcast, OUI_WPA_TKIP, 4);
                pRsnie->ucount = 1;
                NdisMoveMemory(pRsnie->ucast[0].oui, OUI_WPA_TKIP, 4);
                Rsnie_size = sizeof(RSNIE);
                break;

            case Ndis802_11Encryption3Enabled:
                NdisMoveMemory(pRsnie->mcast, OUI_WPA_CCMP, 4);
                pRsnie->ucount = 1;
                NdisMoveMemory(pRsnie->ucast[0].oui, OUI_WPA_CCMP, 4);
                Rsnie_size = sizeof(RSNIE);
                break;

            case Ndis802_11Encryption4Enabled:
                NdisMoveMemory(pRsnie->mcast, OUI_WPA_TKIP, 4);
                pRsnie->ucount = 1;
                NdisMoveMemory(pRsnie->ucast[0].oui, OUI_WPA_TKIP, 4);
                //NdisMoveMemory(pRsnie->ucast[0].oui + 4, OUI_WPA_CCMP, 4);
                Rsnie_size = sizeof(RSNIE) ;
                break;
        }

        pRsnie_auth = (RSNIE_AUTH*)((PUCHAR)pRsnie + Rsnie_size);

        switch (AuthMode)
        {
            case Ndis802_11AuthModeWPA:
            case Ndis802_11AuthModeWPA1WPA2:
                pRsnie_auth->acount = 1;
                NdisMoveMemory(pRsnie_auth->auth[0].oui, OUI_WPA_WEP40, 4);
                break;

            case Ndis802_11AuthModeWPAPSK:
            case Ndis802_11AuthModeWPA1PSKWPA2PSK:
                pRsnie_auth->acount = 1;
                NdisMoveMemory(pRsnie_auth->auth[0].oui, OUI_WPA_TKIP, 4);
                break;
        }

        pRSN_Cap = (RSN_CAPABILITIES*)((PUCHAR)pRsnie_auth + sizeof(RSNIE_AUTH));

//        pAd->ApCfg.RSNIE_Len[0] = Rsnie_size + sizeof(RSNIE_AUTH) + sizeof(RSN_CAPABILITIES);
        if ((AuthMode >= Ndis802_11AuthModeWPA2) &&
            (AuthMode != Ndis802_11AuthModeWPA1PSKWPA2PSK)) /* ~~sample, bug fix, 2006/10/13 */
        {
            pAd->ApCfg.RSNIE_Len[0] = Rsnie_size + sizeof(RSNIE_AUTH) + sizeof(RSN_CAPABILITIES);
        }
        else
            pAd->ApCfg.RSNIE_Len[0] = Rsnie_size + sizeof(RSNIE_AUTH);
        /* End of if */
    }

    // For WPA2, RSN_IE=48, if WPA1WPA2/WPAPSKWPA2PSK mix mode, we store RSN_IE in RSN_IE[1] else RSNIE[0]
    if ((AuthMode == Ndis802_11AuthModeWPA2) || (AuthMode == Ndis802_11AuthModeWPA2PSK)
        || (AuthMode == Ndis802_11AuthModeWPA1WPA2) || (AuthMode == Ndis802_11AuthModeWPA1PSKWPA2PSK))
    {
        RSNIE2              *pRsnie2;
        RSNIE_AUTH          *pRsnie_auth2;
        RSN_CAPABILITIES    *pRSN_Cap;
        UCHAR               Rsnie_size = 0;

        if ((AuthMode == Ndis802_11AuthModeWPA1WPA2) || (AuthMode == Ndis802_11AuthModeWPA1PSKWPA2PSK))
            pRsnie2 = (RSNIE2*)pAd->ApCfg.RSN_IE[1];
        else
            pRsnie2 = (RSNIE2*)pAd->ApCfg.RSN_IE[0];

        pRsnie2->version = 1;

        switch (WepStatus)
        {
            case Ndis802_11Encryption2Enabled:
                NdisMoveMemory(pRsnie2->mcast, OUI_WPA2_TKIP, 4);
                pRsnie2->ucount = 1;
                NdisMoveMemory(pRsnie2->ucast[0].oui, OUI_WPA2_TKIP, 4);
                Rsnie_size = sizeof(RSNIE2);
                break;

            case Ndis802_11Encryption3Enabled:
                NdisMoveMemory(pRsnie2->mcast, OUI_WPA2_CCMP, 4);
                pRsnie2->ucount = 1;
                NdisMoveMemory(pRsnie2->ucast[0].oui, OUI_WPA2_CCMP, 4);
                Rsnie_size = sizeof(RSNIE2);
                break;

            case Ndis802_11Encryption4Enabled:
                NdisMoveMemory(pRsnie2->mcast, OUI_WPA2_TKIP, 4);
                pRsnie2->ucount = 2;
                NdisMoveMemory(pRsnie2->ucast[0].oui, OUI_WPA2_TKIP, 4);
                NdisMoveMemory(pRsnie2->ucast[0].oui + 4, OUI_WPA2_CCMP, 4);
                Rsnie_size = sizeof(RSNIE2) + 4;
                break;
        }

        pRsnie_auth2 = (RSNIE_AUTH*)((PUCHAR)pRsnie2 + Rsnie_size);

        switch (AuthMode)
        {
            case Ndis802_11AuthModeWPA2:
            case Ndis802_11AuthModeWPA1WPA2:
                pRsnie_auth2->acount = 1;
                NdisMoveMemory(pRsnie_auth2->auth[0].oui, OUI_WPA2_WEP40, 4);
                break;

            case Ndis802_11AuthModeWPA2PSK:
            case Ndis802_11AuthModeWPA1PSKWPA2PSK:
                pRsnie_auth2->acount = 1;
                NdisMoveMemory(pRsnie_auth2->auth[0].oui, OUI_WPA2_TKIP, 4);
                break;
        }

        pRSN_Cap = (RSN_CAPABILITIES*)((PUCHAR)pRsnie_auth2 + sizeof(RSNIE_AUTH));
        pRSN_Cap->field.Pre_Auth = (pAd->ApCfg.bPreAuth == TRUE) ? 1 : 0;

        if ((AuthMode == Ndis802_11AuthModeWPA1WPA2) || (AuthMode == Ndis802_11AuthModeWPA1PSKWPA2PSK))
            pAd->ApCfg.RSNIE_Len[1] = Rsnie_size + sizeof(RSNIE_AUTH) + sizeof(RSN_CAPABILITIES);
        else
            pAd->ApCfg.RSNIE_Len[0] = Rsnie_size + sizeof(RSNIE_AUTH) + sizeof(RSN_CAPABILITIES);
    }

    DBGPRINT(RT_DEBUG_TRACE,("RTMPMakeRSNIE : RSNIE_Len = %d\n", pAd->ApCfg.RSNIE_Len[0]));
}

VOID    APToWirelessSta(
    IN  PRTMP_ADAPTER   pAd,
    IN  USHORT          Aid,
    IN  PUCHAR          pHeader802_3,
    IN  UINT            HdrLen,
    IN  PUCHAR          pData,
    IN  UINT            DataLen)
{
    PNDIS_PACKET    pPacket;
//    PNDIS_BUFFER    pBuffer;
    NDIS_STATUS     Status;
    UCHAR QueIdx;
//    UCHAR           *pVirtualAddress;
    do {
        // build a NDIS packet
        Status = RTMPAllocateNdisPacket(pAd, &pPacket, pHeader802_3, HdrLen, pData, DataLen);
        if (Status != NDIS_STATUS_SUCCESS)
            break;


        DBGPRINT(RT_DEBUG_INFO,("APToWirelessSta - len=%d, AID=%d, DA=%02x:%02x:%02x:%02x:%02x:%02x\n",
            HdrLen+DataLen, Aid, pHeader802_3[0],pHeader802_3[1],pHeader802_3[2],pHeader802_3[3],pHeader802_3[4],pHeader802_3[5]));

        // send out the packet
        RTMP_SET_PACKET_WDS(pPacket, (UCHAR)Aid);
        APSendPacket(pAd, pPacket); // send to one of TX queue
        for(QueIdx = 0 ; QueIdx< 4; QueIdx++)
        	RTMPDeQueuePacket(pAd,QueIdx);     // Dequeue outgoing frames from TxSwQueue0..3 queue and process it

	// Kick bulk out
	RTUSBKickBulkOut(pAd);
    } while (FALSE);
}


