#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <stdbool.h>
#include <syslog.h>
#include <stdlib.h>
#include <getopt.h>
#include "common.h"
#include "ctl.h"
#include "dms.h"
#include "fms.h"
#include "lite-qmi-dms.h"
#include <QmuxTransport.h>
#include <QmiService.h>
#include <CtlService.h>
#include <QmiSyncObject.h>

#include <locale.h>
#include <sys/time.h>
#include "msgid.h"

#include "dev_util.h"
#include "qmux_util.h"
#include "str_util.h"

extern void dms_indication_handler(uint16_t msgid, uint8_t *msg, uint32_t rlen);

extern void GetMsisdn(QmiService* pQmiService);
extern void GetFirmwareInformation(QmiService* pQmiService);
extern void GetSerialNumber(QmiService* pQmiService);
extern void GetModelID(QmiService* pQmiService);
extern void GetFsn(QmiService* pQmiService);
extern void GetDeviceCapabilities(QmiService* pQmiService);
extern void GetBandCapability(QmiService* pQmiService);
extern void GetManufacturer(QmiService* pQmiService);
extern void GetSwiHostDevInfo(QmiService* pQmiService);
extern void SetSwiHostDevInfo(QmiService* pQmiService, pack_dms_SLQSSwiSetHostDevInfo_t *pSwiSetHostDevInfo);
extern void SetEventReport(QmiService* pQmiService, pack_dms_SetEventReport_t *pSetEventReport);
extern void GetStoredImages(QmiService * pQmiService);
extern void DeleteStoredImage(QmiService * pQmiService, pack_fms_DeleteStoredImage_t * pDeleteStoredImage);
extern void SetPower(QmiService* pQmiService, pack_dms_SetPower_t *pSetPower);
extern void GetPower(QmiService* pQmiService);

static CtlService s_CtlService;
static QmiService s_DmsService;
static QmuxTransport s_Transport;

//////////////////////////////////////////////////////////
static void PrintPrompt();
static void PrintSetSwiHostDevInfo();
static void PrintSetEventReport();
static void PrintSetPower();

enum AppStateType g_AppState = AppUninitialized;

#define DEVICE_PATH_MAX 256
char g_DevicePath[DEVICE_PATH_MAX]={0};

bool g_PrintVersion = false;
int g_mode = QMUX_INTERFACE_UNKNOWN;

#define APPNAME "lite-qmi-dms"
#define VERSION "1.0.2111.0"

///////////////////////////////////////////////////
static void DmsIndicationCallback(uint8_t* qmiPacket, uint16_t qmiPacketSize, void* pIndicationCallbackContext)
{
	(void)pIndicationCallbackContext;

	unpack_qmi_t rsp_ctx;
	printf (ANSI_COLOR_YELLOW);
	printf("<< receiving dms indication: %s\n", helper_get_resp_ctx(eDMS, qmiPacket, qmiPacketSize, &rsp_ctx));
	printf("msgid 0x%x, type:%x\n", rsp_ctx.msgid,rsp_ctx.type);
	dms_indication_handler(rsp_ctx.msgid, qmiPacket,  qmiPacketSize);
	printf (ANSI_COLOR_RESET);
}

static void PrintPrompt()
{
    printf("\n     1.  Get Model ID"
           "\n     2.  Get FSN"
           "\n     3.  Get Device Capabilities"
           "\n     4.  Get Band Capabilities"
           "\n     5.  Get Device Manufacturer"
           "\n     6.  Get MSISDN"
           "\n     7.  Get Firmware Information"
           "\n     8.  Get MEID/IMEI"
           "\n     9.  Get SWI Host Dev Info"
           "\n     10. Set SWI Host Dev Info"
           "\n     11. Set Event Report"
           "\n     12. Get Stored Images"
           "\n     13. Delete Stored Image"
           "\n     14. Set Power"
           "\n     15. Get Power"
		"\n     (q)uit to exit: \n");

    fflush(stdout);
}

static void PrintDeleteStoredImage()
{
    fprintf(stderr, "\nPlease enter information of the image to be deleted:\n"\
                    "\nType(0: Modem, 1: PRI)= ImageID= BuildID=\n"\
                    "\nex. Type=1 ImageID=002.007_000 BuildID=01.08.04.00_?\n"\
                    "  Or Press<Enter> to go to previous menu:\n");
    g_AppState = AppDeleteStoredImage;
}

static void PrintSetSwiHostDevInfo()
{
    fprintf( stderr, "\nPlease enter SWI Host Dev Info [all params optional]:\n"\
                     "\nManufacturer=,Model=,Version=,PlasmaID=,HostID=,Instance=\n"\
                     "\nex. Manufacturer=Sierra Wireless,Model=SDX55,Version=05.05.58.00,PlasmaID=PL110,HostID=SWI\n"\
                     "  Or Press<Enter> to go to previous menu:\n");
    g_AppState = AppSetSwiHostDevInfo;
}

static void PrintSetEventReport()
{
    fprintf( stderr, "\nPlease enter Set Event Report Mode(0: reset 1: set 2: no change):\n"\
                     "  Or Press<Enter> to go to previous menu:\n");
    g_AppState = AppSetEventReport;
}

static void PrintSetPower()
{
    fprintf( stderr, "\nPlease enter Power Mode(0- Online 1- Low power 2- Factory Test mode 3- Offline 4- Resetting 5- Shutting down\n"\
                     "6- Persistent low power 7- Mode-only low power 8- Conducting network test for GSM/WCDMA 9- Camp only\n"\
                     "  Or Press<Enter> to go to previous menu:\n");
    g_AppState = AppSetPower;
}

static void ExecuteDmsTestCaseSelection(void)
{
	char str_return[MAX_USER_INPUT_SIZE];
	PrintPrompt();
	
	while (1)
	{
		fflush(stdin);
		memset (str_return,0,MAX_USER_INPUT_SIZE);

		fgets(str_return,MAX_USER_INPUT_SIZE,stdin);

		switch(g_AppState)
		{
			case AppCommandExecuted:
				g_AppState = AppRunning;
				PrintPrompt();
			break;
			default:
			if (!strcmp(str_return, "\n"))
			{
				g_AppState = AppRunning;
				PrintPrompt();
			}
			break;
		}

		size_t s = strlen(str_return);
		if (s > 0)
			str_return[s - 1] = 0;
		else
			str_return[0] = 0;

		if (!strcmp(str_return, "q"))
		{
			printf("quitting...\n");
			break;
		} 
        else if (!strcmp(str_return, "d"))
		{
			PrintPrompt();
		}
        else if (g_AppState == AppDeleteStoredImage)
        {
            // Type=,ImageID=,BuildID=
            pack_fms_DeleteStoredImage_t packfmsDeleteStoredImage;
            memset(&packfmsDeleteStoredImage, 0, sizeof(pack_fms_DeleteStoredImage_t));

            sscanf(str_return, "Type=%hhu ImageID=%99s BuildID=%99s", &packfmsDeleteStoredImage.ImageInfo.imageType,
                packfmsDeleteStoredImage.ImageInfo.imageId, packfmsDeleteStoredImage.ImageInfo.buildId);
            packfmsDeleteStoredImage.ImageInfo.buildIdLength = (uint8_t)strlen((char *)packfmsDeleteStoredImage.ImageInfo.buildId);

            DeleteStoredImage(&s_DmsService, &packfmsDeleteStoredImage);

            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppSetSwiHostDevInfo)
        {
            //Manufacturer=,Model=,Version=,PlasmaID=,HostID=,Instance=
            uint8_t inst = 1;
	        pack_dms_SLQSSwiSetHostDevInfo_t packdmsSLQSSwiSetHostDevInfo;
            memset(&packdmsSLQSSwiSetHostDevInfo, 0, sizeof(pack_dms_SLQSSwiSetHostDevInfo_t));
            
            char *pChar  = strstr(str_return, "Manufacturer=");
            char *pCharEnd = NULL;
            if (pChar)
            {
                pChar += 13;
                pCharEnd = strchr (pChar, ',');
                if (pCharEnd)
					StrNCpy(packdmsSLQSSwiSetHostDevInfo.manString, pChar, pCharEnd - pChar);
            }
            pChar = strstr(str_return, "Model=");
            if (pChar)
            {
                pChar += 6;
                pCharEnd = strchr (pChar, ',');
                if (pCharEnd)
					StrNCpy(packdmsSLQSSwiSetHostDevInfo.modelString, pChar, pCharEnd - pChar);
            }
            pChar = strstr(str_return, "Version=");
            if (pChar)
            {
                pChar += 8;
                pCharEnd = strchr (pChar, ',');
                if (pCharEnd)
					StrNCpy(packdmsSLQSSwiSetHostDevInfo.swVerString, pChar, pCharEnd - pChar);
            }
            pChar = strstr(str_return, "PlasmaID=");
            if (pChar)
            {
                pChar += 9;
                pCharEnd = strchr (pChar, ',');
                if (pCharEnd)
					StrNCpy(packdmsSLQSSwiSetHostDevInfo.plasmaIDString, pChar, pCharEnd - pChar);
            }
            pChar = strstr(str_return, "HostID=");
            if (pChar)
            {
                pChar += 7;
                pCharEnd = strchr (pChar, ',');
                if (pCharEnd)
					StrNCpy(packdmsSLQSSwiSetHostDevInfo.hostID, pChar, pCharEnd - pChar);
            }
            pChar = strstr(str_return, "Instance=");
            if (pChar)
            {
                pChar += 9;
                inst = atoi( pChar ); 
                packdmsSLQSSwiSetHostDevInfo.pInstance = &inst;
            }

            SetSwiHostDevInfo(&s_DmsService, &packdmsSLQSSwiSetHostDevInfo);

            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppSetEventReport)
        {
            pack_dms_SetEventReport_t sSetEventReport;
            sSetEventReport.mode = 0xFF;
            sSetEventReport.mode = atoi(str_return);

            if (sSetEventReport.mode > 2)
               	printf("\nInvalid argument\n");
            else
                SetEventReport (&s_DmsService, &sSetEventReport);

            g_AppState = AppCommandExecuted;
        }
        else if (g_AppState == AppSetPower)
        {
            pack_dms_SetPower_t sSetPower;
            sSetPower.mode = 0xFF;
            sSetPower.mode = atoi(str_return);

            if (sSetPower.mode > 9)
               	printf("\nInvalid argument\n");
            else
                SetPower (&s_DmsService, &sSetPower);

            g_AppState = AppCommandExecuted;
        }
		else if (g_AppState == AppRunning)
        {

            if (!strcmp(str_return, "1") || !strcmp(str_return, "Get Model ID"))
            {            
                GetModelID(&s_DmsService);
            }
            else if (!strcmp(str_return, "2") || !strcmp(str_return, "Get FSN"))
            {            
                GetFsn(&s_DmsService);
            }
            else if (!strcmp(str_return, "3") || !strcmp(str_return, "Get Device Capabilities"))
            { 
                GetDeviceCapabilities(&s_DmsService);
            }
            else if (!strcmp(str_return, "4") || !strcmp(str_return, "Get Band Capabilities"))
            { 
                GetBandCapability(&s_DmsService);
            }
            else if (!strcmp(str_return, "5") || !strcmp(str_return, "Get Device Manufacturer"))
            {
                GetManufacturer(&s_DmsService);
            }
            else if (!strcmp(str_return, "6") || !strcmp(str_return, "Get MSISDN"))
            {
                GetMsisdn(&s_DmsService);
            }
            else if (!strcmp(str_return, "7") || !strcmp(str_return, "Get Firmware Information"))
            {
                GetFirmwareInformation(&s_DmsService);
            }
            else if (!strcmp(str_return, "8") || !strcmp(str_return, "Get MEID/IMEI"))
            {
                GetSerialNumber(&s_DmsService);
            }
			else if (!strcmp(str_return, "9") || !strcmp(str_return, "Get SWI Host Dev Info"))
			{
				GetSwiHostDevInfo(&s_DmsService);
			}			
			else if (!strcmp(str_return, "10") || !strcmp(str_return, "Set SWI Host Dev Info"))
			{
                PrintSetSwiHostDevInfo();
			}
			else if (!strcmp(str_return, "11") || !strcmp(str_return, "Set Event Report"))
			{
                PrintSetEventReport();
			}
            else if (!strcmp(str_return, "12") || !strcmp(str_return, "Get Stored Images"))
            {
                GetStoredImages(&s_DmsService);
            }
            else if (!strcmp(str_return, "13") || !strcmp(str_return, "Delete Stored Image"))
            {
                PrintDeleteStoredImage();
            }
			else if (!strcmp(str_return, "14") || !strcmp(str_return, "Set Power"))
			{
				PrintSetPower();
			}
			else if (!strcmp(str_return, "15") || !strcmp(str_return, "Get Power"))
			{
				GetPower(&s_DmsService);

				g_AppState = AppCommandExecuted;
			}
		}
	}

	return;
}

void PrintUsageManual()
{
    printf( "\r\n" );
    printf( "App usage: \r\n\r\n" );
    printf( "  -d  --device \n" );
    printf( "        absolute path to qmux device\n\n" );
    printf( "  -m  --mbim \n" );
    printf( "        Device is an MBIM interface (defaults to direct QMUX interface)\n\n" );
	printf( "  -q  --qmi\n");
	printf( "        Use direct QMUX interface (QMI over PCIe, or RmNet over USB)\n\n");
	printf( "  -r  --router\n");
	printf( "        Use QMUX Router\n\n");
	printf( "  -h  --help  \n" );
    printf( "        This option prints the usage instructions.\n\n" );
    printf( "  -V  --version  \n" );
    printf( "        This option prints app version.\n\n" );
}

const char * const s_ShortOptions = "d:hmqrV";

const struct option s_LongOptions[] = {
    {"version",  0, NULL, 'V'},
    {"help",   0, NULL, 'h'},
    {"device", 1, NULL, 'd'},
    {"mbim",   0, NULL, 'm'},
    {"qmi",    0, NULL, 'q'},
	{"router", 0, NULL, 'r'},
	{NULL,     0, NULL,  0 }       /* End of list */
};

static void ParseCommandLine( int argc, char **argv)
{
    int next_option;
    
    /* Parse the command line before doing anything else */
    do
    {
        /* Read the next option until there are no more */
        next_option = getopt_long( argc, argv,
                                   s_ShortOptions,
                                   s_LongOptions, NULL );

        switch( next_option )
        {
            case 'V':
                /* Print usage information */
                g_PrintVersion = true;
                break;
            case 'h':
                /* Print usage information */
                PrintUsageManual();
                exit (0);
                break;
            case 'd':
				StrCpy(g_DevicePath, optarg);
                break;
            case 'm':
                if (g_mode == QMUX_INTERFACE_UNKNOWN)
                    g_mode = QMUX_INTERFACE_MBIM;
                break;
            case 'q':
                if (g_mode == QMUX_INTERFACE_UNKNOWN)
                    g_mode = QMUX_INTERFACE_DIRECT;
                break;
			case 'r':
				if (g_mode == QMUX_INTERFACE_UNKNOWN)
					g_mode = QMUX_INTERFACE_ROUTER;
				break;
			case -1:
                /* Done with options list */
                break;
            default:
                exit(EXIT_FAILURE);
                break;
        }
    }
    while( next_option != -1 );
}

int main(int argc, char **argv)
{
	int ret = FAILURE;
    g_PrintVersion = false;

    ParseCommandLine(argc, argv);
    if (g_PrintVersion)
    {
        printf("\n%s v%s\n\n", APPNAME, VERSION);
       	if (argc == 2)
		    return 0;
    }

    if (OpenTransport(&s_Transport, g_DevicePath, sizeof(g_DevicePath), &g_mode, NULL,
		true, PrintUsageManual, argv[0], false) != 0)
		return 0;

	if (CtlService_Initialize(&s_CtlService, &s_Transport) != SUCCESS)
		printf("CtlService_Initialize failed\n");
	else
	{
		memset(&s_DmsService, 0, sizeof(QmiService));

        // We use the Ctl service to initialize the regular services because it knows how to 
        // acquire client IDs for them.

        // Use a do/while for easy exit. We have to clean up.
        do
        {
            // Infrastructure is now ready. Let's create some regular QMI services.

            ret = CtlService_InitializeRegularService(&s_CtlService, &s_DmsService, eDMS, DmsIndicationCallback, NULL);
            if (ret != SUCCESS)
            {
                printf("InitializeRegularService eDMS failed.\n");
                break;
            }
            g_AppState = AppRunning;
            
            ExecuteDmsTestCaseSelection();

        } while (0);		

        // Shut down.
        g_AppState = AppShuttingDown;

        CtlService_ShutDownRegularService(&s_CtlService, &s_DmsService);

        CtlService_ShutDown(&s_CtlService);
    }

	QmuxTransport_ShutDown(&s_Transport);
	
	return ret;
}
